package units

import (
	"fmt"
	"math"
	"testing"

	"github.com/stretchr/testify/assert"
)

// conversionTest struct defines test data for a conversion, where a value of 1.0 in
// the `from` unit name is expected to equal `val` when converted to the `to` unit name
type conversionTest struct {
	from string
	to   string
	val  string
}

var convTests = []conversionTest{
	{"yard", "millimeter", "914.4"},
	{"yard", "decimeter", "9.144"},
	{"yard", "attometer", "914399999999999872"},
	{"yard", "furlong", "0.00454545"},
	{"yard", "micrometer", "914400"},
	{"yard", "foot", "3"},
	{"yard", "picometer", "914400000000"},
	{"yard", "terameter", "0.0000000000009144"},
	{"yard", "decameter", "0.09144"},
	{"yard", "mile", "0.000568182"},
	{"yard", "league", "0.0001893939"},
	{"yard", "kilometer", "0.0009144"},
	{"yard", "petameter", "0.0000000000000009144"},
	{"yard", "hectometer", "0.009144"},
	{"yard", "nanometer", "914400000"},
	{"yard", "angstrom", "9144000000"},
	{"yard", "megameter", "0.0000009144"},
	{"yard", "gigameter", "0.0000000009144"},
	{"yard", "centimeter", "91.44"},
	{"yard", "exameter", "0.0000000000000000009144"},
	{"yard", "inch", "36"},
	{"yard", "femtometer", "914399999999999.875"},
	{"yard", "meter", "0.9144"},
	{"millimeter", "yard", "0.001093613"},
	{"millimeter", "decimeter", "0.01"},
	{"millimeter", "attometer", "1000000000000000"},
	{"millimeter", "furlong", "0.00000497097"},
	{"millimeter", "micrometer", "1000"},
	{"millimeter", "foot", "0.00328084"},
	{"millimeter", "picometer", "1000000000"},
	{"millimeter", "terameter", "0.000000000000001"},
	{"millimeter", "decameter", "0.0001"},
	{"millimeter", "mile", "0.000000621371"},
	{"millimeter", "league", "0.0000002071237"},
	{"millimeter", "kilometer", "0.000001"},
	{"millimeter", "petameter", "0.000000000000000001"},
	{"millimeter", "hectometer", "0.00001"},
	{"millimeter", "nanometer", "1000000"},
	{"millimeter", "angstrom", "10000000"},
	{"millimeter", "megameter", "0.000000001"},
	{"millimeter", "gigameter", "0.000000000001"},
	{"millimeter", "centimeter", "0.1"},
	{"millimeter", "exameter", "0.000000000000000000001"},
	{"millimeter", "inch", "0.0393701"},
	{"millimeter", "femtometer", "1000000000000"},
	{"millimeter", "meter", "0.001"},
	{"decimeter", "yard", "0.1093613"},
	{"decimeter", "millimeter", "100"},
	{"decimeter", "attometer", "100000000000000000"},
	{"decimeter", "furlong", "0.000497097"},
	{"decimeter", "micrometer", "100000"},
	{"decimeter", "foot", "0.328084"},
	{"decimeter", "picometer", "100000000000"},
	{"decimeter", "terameter", "0.0000000000001"},
	{"decimeter", "decameter", "0.01"},
	{"decimeter", "mile", "0.0000621371"},
	{"decimeter", "league", "0.00002071237"},
	{"decimeter", "kilometer", "0.0001"},
	{"decimeter", "petameter", "0.0000000000000001"},
	{"decimeter", "hectometer", "0.001"},
	{"decimeter", "nanometer", "100000000"},
	{"decimeter", "angstrom", "1000000000"},
	{"decimeter", "megameter", "0.0000001"},
	{"decimeter", "gigameter", "0.0000000001"},
	{"decimeter", "centimeter", "10"},
	{"decimeter", "exameter", "0.0000000000000000001"},
	{"decimeter", "inch", "3.937008"},
	{"decimeter", "femtometer", "100000000000000"},
	{"decimeter", "meter", "0.1"},
	{"attometer", "yard", "0.000000000000000001093613"},
	{"attometer", "millimeter", "0.000000000000001"},
	{"attometer", "decimeter", "0.00000000000000001"},
	{"attometer", "furlong", "0.00000000000000000000497097"},
	{"attometer", "micrometer", "0.000000000001"},
	{"attometer", "foot", "0.00000000000000000328084"},
	{"attometer", "picometer", "0.000001"},
	{"attometer", "terameter", "0.000000000000000000000000000001"},
	{"attometer", "decameter", "0.0000000000000000001"},
	{"attometer", "mile", "0.000000000000000000000621371"},
	{"attometer", "league", "0.0000000000000000000002071237"},
	{"attometer", "kilometer", "0.000000000000000000001"},
	{"attometer", "petameter", "0.000000000000000000000000000000001"},
	{"attometer", "hectometer", "0.00000000000000000001"},
	{"attometer", "nanometer", "0.000000001"},
	{"attometer", "angstrom", "0.00000001"},
	{"attometer", "megameter", "0.000000000000000000000001"},
	{"attometer", "gigameter", "0.000000000000000000000000001"},
	{"attometer", "centimeter", "0.0000000000000001"},
	{"attometer", "exameter", "0.000000000000000000000000000000000001"},
	{"attometer", "inch", "0.0000000000000000393701"},
	{"attometer", "femtometer", "0.001"},
	{"attometer", "meter", "0.000000000000000001"},
	{"furlong", "yard", "220"},
	{"furlong", "millimeter", "201168"},
	{"furlong", "decimeter", "2011.68"},
	{"furlong", "attometer", "201168000000000000000"},
	{"furlong", "micrometer", "201168000"},
	{"furlong", "foot", "660"},
	{"furlong", "picometer", "201168000000000"},
	{"furlong", "terameter", "0.000000000201168"},
	{"furlong", "decameter", "20.1168"},
	{"furlong", "mile", "0.125"},
	{"furlong", "league", "0.0416667"},
	{"furlong", "kilometer", "0.201168"},
	{"furlong", "petameter", "0.000000000000201168"},
	{"furlong", "hectometer", "2.01168"},
	{"furlong", "nanometer", "201168000000"},
	{"furlong", "angstrom", "2011680000000"},
	{"furlong", "megameter", "0.000201168"},
	{"furlong", "gigameter", "0.000000201168"},
	{"furlong", "centimeter", "20116.8"},
	{"furlong", "exameter", "0.000000000000000201168"},
	{"furlong", "inch", "7920"},
	{"furlong", "femtometer", "201168000000000000"},
	{"furlong", "meter", "201.168"},
	{"micrometer", "yard", "0.000001093613"},
	{"micrometer", "millimeter", "0.001"},
	{"micrometer", "decimeter", "0.00001"},
	{"micrometer", "attometer", "999999999999.999878"},
	{"micrometer", "furlong", "0.00000000497097"},
	{"micrometer", "foot", "0.00000328084"},
	{"micrometer", "picometer", "1000000"},
	{"micrometer", "terameter", "0.000000000000000001"},
	{"micrometer", "decameter", "0.0000001"},
	{"micrometer", "mile", "0.000000000621371"},
	{"micrometer", "league", "0.0000000002071237"},
	{"micrometer", "kilometer", "0.000000001"},
	{"micrometer", "petameter", "0.000000000000000000001"},
	{"micrometer", "hectometer", "0.00000001"},
	{"micrometer", "nanometer", "1000"},
	{"micrometer", "angstrom", "10000"},
	{"micrometer", "megameter", "0.000000000001"},
	{"micrometer", "gigameter", "0.000000000000001"},
	{"micrometer", "centimeter", "0.0001"},
	{"micrometer", "exameter", "0.000000000000000000000001"},
	{"micrometer", "inch", "0.0000393701"},
	{"micrometer", "femtometer", "1000000000"},
	{"micrometer", "meter", "0.000001"},
	{"foot", "yard", "0.333333"},
	{"foot", "millimeter", "304.8"},
	{"foot", "decimeter", "3.048"},
	{"foot", "attometer", "304800000000000000"},
	{"foot", "furlong", "0.001515152"},
	{"foot", "micrometer", "304800"},
	{"foot", "picometer", "304800000000"},
	{"foot", "terameter", "0.0000000000003048"},
	{"foot", "decameter", "0.03048"},
	{"foot", "mile", "0.0001893939"},
	{"foot", "league", "0.0000631313"},
	{"foot", "kilometer", "0.0003048"},
	{"foot", "petameter", "0.0000000000000003048"},
	{"foot", "hectometer", "0.003048"},
	{"foot", "nanometer", "304800000"},
	{"foot", "angstrom", "3048000000"},
	{"foot", "megameter", "0.0000003048"},
	{"foot", "gigameter", "0.0000000003048"},
	{"foot", "centimeter", "30.48"},
	{"foot", "exameter", "0.0000000000000000003048"},
	{"foot", "inch", "12"},
	{"foot", "femtometer", "304800000000000"},
	{"foot", "meter", "0.3048"},
	{"picometer", "yard", "0.000000000001093613"},
	{"picometer", "millimeter", "0.000000001"},
	{"picometer", "decimeter", "0.00000000001"},
	{"picometer", "attometer", "1000000"},
	{"picometer", "furlong", "0.00000000000000497097"},
	{"picometer", "micrometer", "0.000001"},
	{"picometer", "foot", "0.00000000000328084"},
	{"picometer", "terameter", "0.000000000000000000000001"},
	{"picometer", "decameter", "0.0000000000001"},
	{"picometer", "mile", "0.000000000000000621371"},
	{"picometer", "league", "0.0000000000000002071237"},
	{"picometer", "kilometer", "0.000000000000001"},
	{"picometer", "petameter", "0.000000000000000000000000001"},
	{"picometer", "hectometer", "0.00000000000001"},
	{"picometer", "nanometer", "0.001"},
	{"picometer", "angstrom", "0.01"},
	{"picometer", "megameter", "0.000000000000000001"},
	{"picometer", "gigameter", "0.000000000000000000001"},
	{"picometer", "centimeter", "0.0000000001"},
	{"picometer", "exameter", "0.000000000000000000000000000001"},
	{"picometer", "inch", "0.0000000000393701"},
	{"picometer", "femtometer", "1000"},
	{"picometer", "meter", "0.000000000001"},
	{"terameter", "yard", "1093613298337.707764"},
	{"terameter", "millimeter", "1000000000000000"},
	{"terameter", "decimeter", "10000000000000"},
	{"terameter", "attometer", "999999999999999879147136483328"},
	{"terameter", "furlong", "4970969537.898671"},
	{"terameter", "micrometer", "1000000000000000000"},
	{"terameter", "foot", "3280839895013.123047"},
	{"terameter", "picometer", "999999999999999983222784"},
	{"terameter", "decameter", "100000000000"},
	{"terameter", "mile", "621371192.237334"},
	{"terameter", "league", "207123730.745778"},
	{"terameter", "kilometer", "1000000000"},
	{"terameter", "petameter", "0.001"},
	{"terameter", "hectometer", "10000000000"},
	{"terameter", "nanometer", "1000000000000000000000"},
	{"terameter", "angstrom", "10000000000000000000000"},
	{"terameter", "megameter", "1000000"},
	{"terameter", "gigameter", "1000"},
	{"terameter", "centimeter", "100000000000000"},
	{"terameter", "exameter", "0.000001"},
	{"terameter", "inch", "39370078740157.484375"},
	{"terameter", "femtometer", "999999999999999875848601600"},
	{"terameter", "meter", "1000000000000"},
	{"decameter", "yard", "10.936133"},
	{"decameter", "millimeter", "10000"},
	{"decameter", "decimeter", "100"},
	{"decameter", "attometer", "10000000000000000000"},
	{"decameter", "furlong", "0.0497097"},
	{"decameter", "micrometer", "10000000"},
	{"decameter", "foot", "32.808399"},
	{"decameter", "picometer", "10000000000000"},
	{"decameter", "terameter", "0.00000000001"},
	{"decameter", "mile", "0.00621371"},
	{"decameter", "league", "0.002071237"},
	{"decameter", "kilometer", "0.01"},
	{"decameter", "petameter", "0.00000000000001"},
	{"decameter", "hectometer", "0.1"},
	{"decameter", "nanometer", "10000000000"},
	{"decameter", "angstrom", "100000000000"},
	{"decameter", "megameter", "0.00001"},
	{"decameter", "gigameter", "0.00000001"},
	{"decameter", "centimeter", "1000"},
	{"decameter", "exameter", "0.00000000000000001"},
	{"decameter", "inch", "393.700787"},
	{"decameter", "femtometer", "10000000000000000"},
	{"decameter", "meter", "10"},
	{"mile", "yard", "1760"},
	{"mile", "millimeter", "1609344"},
	{"mile", "decimeter", "16093.44"},
	{"mile", "attometer", "1609344000000000000000"},
	{"mile", "furlong", "8"},
	{"mile", "micrometer", "1609344000"},
	{"mile", "foot", "5280"},
	{"mile", "picometer", "1609344000000000"},
	{"mile", "terameter", "0.000000001609344"},
	{"mile", "decameter", "160.9344"},
	{"mile", "league", "0.333333"},
	{"mile", "kilometer", "1.609344"},
	{"mile", "petameter", "0.000000000001609344"},
	{"mile", "hectometer", "16.09344"},
	{"mile", "nanometer", "1609344000000"},
	{"mile", "angstrom", "16093440000000"},
	{"mile", "megameter", "0.001609344"},
	{"mile", "gigameter", "0.000001609344"},
	{"mile", "centimeter", "160934.4"},
	{"mile", "exameter", "0.000000000000001609344"},
	{"mile", "inch", "63360"},
	{"mile", "femtometer", "1609344000000000000"},
	{"mile", "meter", "1609.344"},
	{"league", "yard", "5280"},
	{"league", "millimeter", "4828032"},
	{"league", "decimeter", "48280.32"},
	{"league", "attometer", "4828032000000000000000"},
	{"league", "furlong", "24"},
	{"league", "micrometer", "4828032000"},
	{"league", "foot", "15840"},
	{"league", "picometer", "4828032000000000"},
	{"league", "terameter", "0.00000000482803"},
	{"league", "decameter", "482.8032"},
	{"league", "mile", "3"},
	{"league", "kilometer", "4.828032"},
	{"league", "petameter", "0.00000000000482803"},
	{"league", "hectometer", "48.28032"},
	{"league", "nanometer", "4828032000000"},
	{"league", "angstrom", "48280320000000"},
	{"league", "megameter", "0.00482803"},
	{"league", "gigameter", "0.00000482803"},
	{"league", "centimeter", "482803.2"},
	{"league", "exameter", "0.00000000000000482803"},
	{"league", "inch", "190080"},
	{"league", "femtometer", "4828032000000000000"},
	{"league", "meter", "4828.032"},
	{"kilometer", "yard", "1093.613298"},
	{"kilometer", "millimeter", "1000000"},
	{"kilometer", "decimeter", "10000"},
	{"kilometer", "attometer", "999999999999999868928"},
	{"kilometer", "furlong", "4.97097"},
	{"kilometer", "micrometer", "1000000000"},
	{"kilometer", "foot", "3280.839895"},
	{"kilometer", "picometer", "1000000000000000"},
	{"kilometer", "terameter", "0.000000001"},
	{"kilometer", "decameter", "100"},
	{"kilometer", "mile", "0.621371"},
	{"kilometer", "league", "0.2071237"},
	{"kilometer", "petameter", "0.000000000001"},
	{"kilometer", "hectometer", "10"},
	{"kilometer", "nanometer", "999999999999.999878"},
	{"kilometer", "angstrom", "10000000000000"},
	{"kilometer", "megameter", "0.001"},
	{"kilometer", "gigameter", "0.000001"},
	{"kilometer", "centimeter", "100000"},
	{"kilometer", "exameter", "0.000000000000001"},
	{"kilometer", "inch", "39370.07874"},
	{"kilometer", "femtometer", "999999999999999872"},
	{"kilometer", "meter", "1000"},
	{"petameter", "yard", "1093613298337707.75"},
	{"petameter", "millimeter", "1000000000000000000"},
	{"petameter", "decimeter", "10000000000000000"},
	{"petameter", "attometer", "999999999999999945575230987042816"},
	{"petameter", "furlong", "4970969537898.671875"},
	{"petameter", "micrometer", "1000000000000000000000"},
	{"petameter", "foot", "3280839895013123"},
	{"petameter", "picometer", "1000000000000000013287555072"},
	{"petameter", "terameter", "1000"},
	{"petameter", "decameter", "100000000000000"},
	{"petameter", "mile", "621371192237.333984"},
	{"petameter", "league", "207123730745.777985"},
	{"petameter", "kilometer", "1000000000000"},
	{"petameter", "hectometer", "10000000000000"},
	{"petameter", "nanometer", "999999999999999983222784"},
	{"petameter", "angstrom", "9999999999999998758486016"},
	{"petameter", "megameter", "1000000000"},
	{"petameter", "gigameter", "1000000"},
	{"petameter", "centimeter", "100000000000000000"},
	{"petameter", "exameter", "0.001"},
	{"petameter", "inch", "39370078740157480"},
	{"petameter", "femtometer", "999999999999999879147136483328"},
	{"petameter", "meter", "1000000000000000"},
	{"hectometer", "yard", "109.36133"},
	{"hectometer", "millimeter", "100000"},
	{"hectometer", "decimeter", "1000"},
	{"hectometer", "attometer", "100000000000000000000"},
	{"hectometer", "furlong", "0.497097"},
	{"hectometer", "micrometer", "100000000"},
	{"hectometer", "foot", "328.08399"},
	{"hectometer", "picometer", "100000000000000"},
	{"hectometer", "terameter", "0.0000000001"},
	{"hectometer", "decameter", "10"},
	{"hectometer", "mile", "0.0621371"},
	{"hectometer", "league", "0.02071237"},
	{"hectometer", "kilometer", "0.1"},
	{"hectometer", "petameter", "0.0000000000001"},
	{"hectometer", "nanometer", "100000000000"},
	{"hectometer", "angstrom", "1000000000000"},
	{"hectometer", "megameter", "0.0001"},
	{"hectometer", "gigameter", "0.0000001"},
	{"hectometer", "centimeter", "10000"},
	{"hectometer", "exameter", "0.0000000000000001"},
	{"hectometer", "inch", "3937.007874"},
	{"hectometer", "femtometer", "100000000000000000"},
	{"hectometer", "meter", "100"},
	{"nanometer", "yard", "0.000000001093613"},
	{"nanometer", "millimeter", "0.000001"},
	{"nanometer", "decimeter", "0.00000001"},
	{"nanometer", "attometer", "1000000000"},
	{"nanometer", "furlong", "0.00000000000497097"},
	{"nanometer", "micrometer", "0.001"},
	{"nanometer", "foot", "0.00000000328084"},
	{"nanometer", "picometer", "1000"},
	{"nanometer", "terameter", "0.000000000000000000001"},
	{"nanometer", "decameter", "0.0000000001"},
	{"nanometer", "mile", "0.000000000000621371"},
	{"nanometer", "league", "0.0000000000002071237"},
	{"nanometer", "kilometer", "0.000000000001"},
	{"nanometer", "petameter", "0.000000000000000000000001"},
	{"nanometer", "hectometer", "0.00000000001"},
	{"nanometer", "angstrom", "10"},
	{"nanometer", "megameter", "0.000000000000001"},
	{"nanometer", "gigameter", "0.000000000000000001"},
	{"nanometer", "centimeter", "0.0000001"},
	{"nanometer", "exameter", "0.000000000000000000000000001"},
	{"nanometer", "inch", "0.0000000393701"},
	{"nanometer", "femtometer", "1000000"},
	{"nanometer", "meter", "0.000000001"},
	{"angstrom", "yard", "0.0000000001093613"},
	{"angstrom", "millimeter", "0.0000001"},
	{"angstrom", "decimeter", "0.000000001"},
	{"angstrom", "attometer", "100000000"},
	{"angstrom", "furlong", "0.000000000000497097"},
	{"angstrom", "micrometer", "0.0001"},
	{"angstrom", "foot", "0.000000000328084"},
	{"angstrom", "picometer", "100"},
	{"angstrom", "terameter", "0.0000000000000000000001"},
	{"angstrom", "decameter", "0.00000000001"},
	{"angstrom", "mile", "0.0000000000000621371"},
	{"angstrom", "league", "0.00000000000002071237"},
	{"angstrom", "kilometer", "0.0000000000001"},
	{"angstrom", "petameter", "0.0000000000000000000000001"},
	{"angstrom", "hectometer", "0.000000000001"},
	{"angstrom", "nanometer", "0.1"},
	{"angstrom", "megameter", "0.0000000000000001"},
	{"angstrom", "gigameter", "0.0000000000000000001"},
	{"angstrom", "centimeter", "0.00000001"},
	{"angstrom", "exameter", "0.0000000000000000000000000001"},
	{"angstrom", "inch", "0.00000000393701"},
	{"angstrom", "femtometer", "100000"},
	{"angstrom", "meter", "0.0000000001"},
	{"megameter", "yard", "1093613.298338"},
	{"megameter", "millimeter", "1000000000"},
	{"megameter", "decimeter", "10000000"},
	{"megameter", "attometer", "999999999999999983222784"},
	{"megameter", "furlong", "4970.969538"},
	{"megameter", "micrometer", "1000000000000"},
	{"megameter", "foot", "3280839.895013"},
	{"megameter", "picometer", "1000000000000000000"},
	{"megameter", "terameter", "0.000001"},
	{"megameter", "decameter", "100000"},
	{"megameter", "mile", "621.371192"},
	{"megameter", "league", "207.123731"},
	{"megameter", "kilometer", "1000"},
	{"megameter", "petameter", "0.000000001"},
	{"megameter", "hectometer", "10000"},
	{"megameter", "nanometer", "1000000000000000"},
	{"megameter", "angstrom", "10000000000000000"},
	{"megameter", "gigameter", "0.001"},
	{"megameter", "centimeter", "100000000"},
	{"megameter", "exameter", "0.000000000001"},
	{"megameter", "inch", "39370078.740157"},
	{"megameter", "femtometer", "999999999999999868928"},
	{"megameter", "meter", "1000000"},
	{"gigameter", "yard", "1093613298.337708"},
	{"gigameter", "millimeter", "1000000000000"},
	{"gigameter", "decimeter", "10000000000"},
	{"gigameter", "attometer", "999999999999999875848601600"},
	{"gigameter", "furlong", "4970969.537899"},
	{"gigameter", "micrometer", "1000000000000000"},
	{"gigameter", "foot", "3280839895.013123"},
	{"gigameter", "picometer", "1000000000000000000000"},
	{"gigameter", "terameter", "0.001"},
	{"gigameter", "decameter", "100000000"},
	{"gigameter", "mile", "621371.192237"},
	{"gigameter", "league", "207123.730746"},
	{"gigameter", "kilometer", "1000000"},
	{"gigameter", "petameter", "0.000001"},
	{"gigameter", "hectometer", "10000000"},
	{"gigameter", "nanometer", "1000000000000000000"},
	{"gigameter", "angstrom", "10000000000000000000"},
	{"gigameter", "megameter", "1000"},
	{"gigameter", "centimeter", "100000000000"},
	{"gigameter", "exameter", "0.000000001"},
	{"gigameter", "inch", "39370078740.157478"},
	{"gigameter", "femtometer", "999999999999999983222784"},
	{"gigameter", "meter", "1000000000"},
	{"centimeter", "yard", "0.01093613"},
	{"centimeter", "millimeter", "10"},
	{"centimeter", "decimeter", "0.1"},
	{"centimeter", "attometer", "10000000000000000"},
	{"centimeter", "furlong", "0.0000497097"},
	{"centimeter", "micrometer", "10000"},
	{"centimeter", "foot", "0.0328084"},
	{"centimeter", "picometer", "10000000000"},
	{"centimeter", "terameter", "0.00000000000001"},
	{"centimeter", "decameter", "0.001"},
	{"centimeter", "mile", "0.00000621371"},
	{"centimeter", "league", "0.000002071237"},
	{"centimeter", "kilometer", "0.00001"},
	{"centimeter", "petameter", "0.00000000000000001"},
	{"centimeter", "hectometer", "0.0001"},
	{"centimeter", "nanometer", "10000000"},
	{"centimeter", "angstrom", "100000000"},
	{"centimeter", "megameter", "0.00000001"},
	{"centimeter", "gigameter", "0.00000000001"},
	{"centimeter", "exameter", "0.00000000000000000001"},
	{"centimeter", "inch", "0.393701"},
	{"centimeter", "femtometer", "10000000000000"},
	{"centimeter", "meter", "0.01"},
	{"exameter", "yard", "1093613298337707776"},
	{"exameter", "millimeter", "1000000000000000000000"},
	{"exameter", "decimeter", "10000000000000000000"},
	{"exameter", "attometer", "999999999999999894846684784341549056"},
	{"exameter", "furlong", "4970969537898672"},
	{"exameter", "micrometer", "999999999999999983222784"},
	{"exameter", "foot", "3280839895013123072"},
	{"exameter", "picometer", "1000000000000000019884624838656"},
	{"exameter", "terameter", "1000000"},
	{"exameter", "decameter", "100000000000000000"},
	{"exameter", "mile", "621371192237334"},
	{"exameter", "league", "207123730745777.96875"},
	{"exameter", "kilometer", "1000000000000000"},
	{"exameter", "petameter", "1000"},
	{"exameter", "hectometer", "10000000000000000"},
	{"exameter", "nanometer", "999999999999999875848601600"},
	{"exameter", "angstrom", "9999999999999999583119736832"},
	{"exameter", "megameter", "1000000000000"},
	{"exameter", "gigameter", "1000000000"},
	{"exameter", "centimeter", "100000000000000000000"},
	{"exameter", "inch", "39370078740157480960"},
	{"exameter", "femtometer", "999999999999999945575230987042816"},
	{"exameter", "meter", "1000000000000000000"},
	{"inch", "yard", "0.02777778"},
	{"inch", "millimeter", "25.4"},
	{"inch", "decimeter", "0.254"},
	{"inch", "attometer", "25399999999999996"},
	{"inch", "furlong", "0.0001262626"},
	{"inch", "micrometer", "25400"},
	{"inch", "foot", "0.0833333"},
	{"inch", "picometer", "25400000000"},
	{"inch", "terameter", "0.0000000000000254"},
	{"inch", "decameter", "0.00254"},
	{"inch", "mile", "0.00001578283"},
	{"inch", "league", "0.00000526094"},
	{"inch", "kilometer", "0.0000254"},
	{"inch", "petameter", "0.0000000000000000254"},
	{"inch", "hectometer", "0.000254"},
	{"inch", "nanometer", "25400000"},
	{"inch", "angstrom", "254000000"},
	{"inch", "megameter", "0.0000000254"},
	{"inch", "gigameter", "0.0000000000254"},
	{"inch", "centimeter", "2.54"},
	{"inch", "exameter", "0.0000000000000000000254"},
	{"inch", "femtometer", "25399999999999.996094"},
	{"inch", "meter", "0.0254"},
	{"femtometer", "yard", "0.000000000000001093613"},
	{"femtometer", "millimeter", "0.000000000001"},
	{"femtometer", "decimeter", "0.00000000000001"},
	{"femtometer", "attometer", "1000"},
	{"femtometer", "furlong", "0.00000000000000000497097"},
	{"femtometer", "micrometer", "0.000000001"},
	{"femtometer", "foot", "0.00000000000000328084"},
	{"femtometer", "picometer", "0.001"},
	{"femtometer", "terameter", "0.000000000000000000000000001"},
	{"femtometer", "decameter", "0.0000000000000001"},
	{"femtometer", "mile", "0.000000000000000000621371"},
	{"femtometer", "league", "0.0000000000000000002071237"},
	{"femtometer", "kilometer", "0.000000000000000001"},
	{"femtometer", "petameter", "0.000000000000000000000000000001"},
	{"femtometer", "hectometer", "0.00000000000000001"},
	{"femtometer", "nanometer", "0.000001"},
	{"femtometer", "angstrom", "0.00001"},
	{"femtometer", "megameter", "0.000000000000000000001"},
	{"femtometer", "gigameter", "0.000000000000000000000001"},
	{"femtometer", "centimeter", "0.0000000000001"},
	{"femtometer", "exameter", "0.000000000000000000000000000000001"},
	{"femtometer", "inch", "0.0000000000000393701"},
	{"femtometer", "meter", "0.000000000000001"},
	{"meter", "yard", "1.093613"},
	{"meter", "millimeter", "1000"},
	{"meter", "decimeter", "10"},
	{"meter", "attometer", "999999999999999872"},
	{"meter", "furlong", "0.00497097"},
	{"meter", "micrometer", "1000000"},
	{"meter", "foot", "3.28084"},
	{"meter", "picometer", "1000000000000"},
	{"meter", "terameter", "0.000000000001"},
	{"meter", "decameter", "0.1"},
	{"meter", "mile", "0.000621371"},
	{"meter", "league", "0.0002071237"},
	{"meter", "kilometer", "0.001"},
	{"meter", "petameter", "0.000000000000001"},
	{"meter", "hectometer", "0.01"},
	{"meter", "nanometer", "1000000000"},
	{"meter", "angstrom", "10000000000"},
	{"meter", "megameter", "0.000001"},
	{"meter", "gigameter", "0.000000001"},
	{"meter", "centimeter", "100"},
	{"meter", "exameter", "0.000000000000000001"},
	{"meter", "inch", "39.370079"},
	{"meter", "femtometer", "999999999999999.875"},
	{"inch of Mercury", "decapascal", "338.638816"},
	{"inch of Mercury", "millibar", "34.554981"},
	{"inch of Mercury", "pound-force per square inch", "0.491154"},
	{"inch of Mercury", "picopascal", "3386388157890000"},
	{"inch of Mercury", "gigapascal", "0.00000338639"},
	{"inch of Mercury", "centipascal", "338638.815789"},
	{"inch of Mercury", "barye", "33863.881579"},
	{"inch of Mercury", "millimeter of Water Column", "3453.154908"},
	{"inch of Mercury", "meter of Mercury", "0.254"},
	{"inch of Mercury", "petapascal", "0.00000000000338639"},
	{"inch of Mercury", "bar", "0.034555"},
	{"inch of Mercury", "terapascal", "0.00000000338639"},
	{"inch of Mercury", "inch of Water Column", "13.608697"},
	{"inch of Mercury", "newton per square meter", "3386.388158"},
	{"inch of Mercury", "millipascal", "3386388.15789"},
	{"inch of Mercury", "exapascal", "0.00000000000000338639"},
	{"inch of Mercury", "attopascal", "3386388157889999536128"},
	{"inch of Mercury", "meter of Water Column", "3.453155"},
	{"inch of Mercury", "kilopascal", "3.386388"},
	{"inch of Mercury", "torr", "25.4"},
	{"inch of Mercury", "micropascal", "3386388157.89"},
	{"inch of Mercury", "femtopascal", "3386388157889999872"},
	{"inch of Mercury", "technical atmosphere", "0.0345315"},
	{"inch of Mercury", "hectopascal", "33.863882"},
	{"inch of Mercury", "nanopascal", "3386388157890"},
	{"inch of Mercury", "millimeter of Mercury", "254"},
	{"inch of Mercury", "standard atmosphere", "0.033421"},
	{"inch of Mercury", "centibar", "3.455498"},
	{"inch of Mercury", "pascal", "3386.388158"},
	{"inch of Mercury", "megapascal", "0.00338639"},
	{"inch of Mercury", "microbar", "34554.981203"},
	{"inch of Mercury", "centimeter of Water Column", "345.315491"},
	{"inch of Mercury", "centimeter of Mercury", "25.4"},
	{"inch of Mercury", "decipascal", "33863.881579"},
	{"decapascal", "inch of Mercury", "0.002952999"},
	{"decapascal", "millibar", "0.1020408"},
	{"decapascal", "pound-force per square inch", "0.001450377"},
	{"decapascal", "picopascal", "10000000000000"},
	{"decapascal", "gigapascal", "0.00000001"},
	{"decapascal", "centipascal", "1000"},
	{"decapascal", "barye", "100"},
	{"decapascal", "millimeter of Water Column", "10.197162"},
	{"decapascal", "meter of Mercury", "0.000750062"},
	{"decapascal", "petapascal", "0.00000000000001"},
	{"decapascal", "bar", "0.0001020408"},
	{"decapascal", "terapascal", "0.00000000001"},
	{"decapascal", "inch of Water Column", "0.0401865"},
	{"decapascal", "newton per square meter", "10"},
	{"decapascal", "millipascal", "10000"},
	{"decapascal", "exapascal", "0.00000000000000001"},
	{"decapascal", "attopascal", "10000000000000000000"},
	{"decapascal", "meter of Water Column", "0.01019716"},
	{"decapascal", "kilopascal", "0.01"},
	{"decapascal", "torr", "0.0750062"},
	{"decapascal", "micropascal", "10000000"},
	{"decapascal", "femtopascal", "10000000000000000"},
	{"decapascal", "technical atmosphere", "0.0001019716"},
	{"decapascal", "hectopascal", "0.1"},
	{"decapascal", "nanopascal", "10000000000"},
	{"decapascal", "millimeter of Mercury", "0.750062"},
	{"decapascal", "standard atmosphere", "0.0000986921"},
	{"decapascal", "centibar", "0.01020408"},
	{"decapascal", "pascal", "10"},
	{"decapascal", "megapascal", "0.00001"},
	{"decapascal", "microbar", "102.040816"},
	{"decapascal", "centimeter of Water Column", "1.019716"},
	{"decapascal", "centimeter of Mercury", "0.0750062"},
	{"decapascal", "decipascal", "100"},
	{"millibar", "inch of Mercury", "0.02893939"},
	{"millibar", "decapascal", "9.8"},
	{"millibar", "pound-force per square inch", "0.0142137"},
	{"millibar", "picopascal", "98000000000000"},
	{"millibar", "gigapascal", "0.000000098"},
	{"millibar", "centipascal", "9800"},
	{"millibar", "barye", "980"},
	{"millibar", "millimeter of Water Column", "99.932189"},
	{"millibar", "meter of Mercury", "0.0073506"},
	{"millibar", "petapascal", "0.000000000000098"},
	{"millibar", "bar", "0.001"},
	{"millibar", "terapascal", "0.000000000098"},
	{"millibar", "inch of Water Column", "0.393827"},
	{"millibar", "newton per square meter", "98"},
	{"millibar", "millipascal", "98000"},
	{"millibar", "exapascal", "0.000000000000000098"},
	{"millibar", "attopascal", "98000000000000000000"},
	{"millibar", "meter of Water Column", "0.0999322"},
	{"millibar", "kilopascal", "0.098"},
	{"millibar", "torr", "0.73506"},
	{"millibar", "micropascal", "98000000"},
	{"millibar", "femtopascal", "98000000000000000"},
	{"millibar", "technical atmosphere", "0.000999322"},
	{"millibar", "hectopascal", "0.98"},
	{"millibar", "nanopascal", "98000000000"},
	{"millibar", "millimeter of Mercury", "7.350604"},
	{"millibar", "standard atmosphere", "0.000967182"},
	{"millibar", "centibar", "0.1"},
	{"millibar", "pascal", "98"},
	{"millibar", "megapascal", "0.000098"},
	{"millibar", "microbar", "1000"},
	{"millibar", "centimeter of Water Column", "9.993219"},
	{"millibar", "centimeter of Mercury", "0.73506"},
	{"millibar", "decipascal", "980"},
	{"pound-force per square inch", "inch of Mercury", "2.036021"},
	{"pound-force per square inch", "decapascal", "689.4757"},
	{"pound-force per square inch", "millibar", "70.354663"},
	{"pound-force per square inch", "picopascal", "6894757000000000"},
	{"pound-force per square inch", "gigapascal", "0.00000689476"},
	{"pound-force per square inch", "centipascal", "689475.7"},
	{"pound-force per square inch", "barye", "68947.57"},
	{"pound-force per square inch", "millimeter of Water Column", "7030.695497"},
	{"pound-force per square inch", "meter of Mercury", "0.517149"},
	{"pound-force per square inch", "petapascal", "0.00000000000689476"},
	{"pound-force per square inch", "bar", "0.0703547"},
	{"pound-force per square inch", "terapascal", "0.00000000689476"},
	{"pound-force per square inch", "inch of Water Column", "27.707591"},
	{"pound-force per square inch", "newton per square meter", "6894.757"},
	{"pound-force per square inch", "millipascal", "6894757"},
	{"pound-force per square inch", "exapascal", "0.00000000000000689476"},
	{"pound-force per square inch", "attopascal", "6894756999999998918656"},
	{"pound-force per square inch", "meter of Water Column", "7.030695"},
	{"pound-force per square inch", "kilopascal", "6.894757"},
	{"pound-force per square inch", "torr", "51.71493"},
	{"pound-force per square inch", "micropascal", "6894757000"},
	{"pound-force per square inch", "femtopascal", "6894756999999998976"},
	{"pound-force per square inch", "technical atmosphere", "0.070307"},
	{"pound-force per square inch", "hectopascal", "68.94757"},
	{"pound-force per square inch", "nanopascal", "6894756999999.999023"},
	{"pound-force per square inch", "millimeter of Mercury", "517.149304"},
	{"pound-force per square inch", "standard atmosphere", "0.0680458"},
	{"pound-force per square inch", "centibar", "7.035466"},
	{"pound-force per square inch", "pascal", "6894.757"},
	{"pound-force per square inch", "megapascal", "0.00689476"},
	{"pound-force per square inch", "microbar", "70354.663265"},
	{"pound-force per square inch", "centimeter of Water Column", "703.06955"},
	{"pound-force per square inch", "centimeter of Mercury", "51.71493"},
	{"pound-force per square inch", "decipascal", "68947.57"},
	{"picopascal", "inch of Mercury", "0.0000000000000002952999"},
	{"picopascal", "decapascal", "0.0000000000001"},
	{"picopascal", "millibar", "0.00000000000001020408"},
	{"picopascal", "pound-force per square inch", "0.0000000000000001450377"},
	{"picopascal", "gigapascal", "0.000000000000000000001"},
	{"picopascal", "centipascal", "0.0000000001"},
	{"picopascal", "barye", "0.00000000001"},
	{"picopascal", "millimeter of Water Column", "0.000000000001019716"},
	{"picopascal", "meter of Mercury", "0.0000000000000000750062"},
	{"picopascal", "petapascal", "0.000000000000000000000000001"},
	{"picopascal", "bar", "0.00000000000000001020408"},
	{"picopascal", "terapascal", "0.000000000000000000000001"},
	{"picopascal", "inch of Water Column", "0.00000000000000401865"},
	{"picopascal", "newton per square meter", "0.000000000001"},
	{"picopascal", "millipascal", "0.000000001"},
	{"picopascal", "exapascal", "0.000000000000000000000000000001"},
	{"picopascal", "attopascal", "1000000"},
	{"picopascal", "meter of Water Column", "0.000000000000001019716"},
	{"picopascal", "kilopascal", "0.000000000000001"},
	{"picopascal", "torr", "0.00000000000000750062"},
	{"picopascal", "micropascal", "0.000001"},
	{"picopascal", "femtopascal", "1000"},
	{"picopascal", "technical atmosphere", "0.00000000000000001019716"},
	{"picopascal", "hectopascal", "0.00000000000001"},
	{"picopascal", "nanopascal", "0.001"},
	{"picopascal", "millimeter of Mercury", "0.0000000000000750062"},
	{"picopascal", "standard atmosphere", "0.00000000000000000986921"},
	{"picopascal", "centibar", "0.000000000000001020408"},
	{"picopascal", "pascal", "0.000000000001"},
	{"picopascal", "megapascal", "0.000000000000000001"},
	{"picopascal", "microbar", "0.00000000001020408"},
	{"picopascal", "centimeter of Water Column", "0.0000000000001019716"},
	{"picopascal", "centimeter of Mercury", "0.00000000000000750062"},
	{"picopascal", "decipascal", "0.00000000001"},
	{"gigapascal", "inch of Mercury", "295299.875081"},
	{"gigapascal", "decapascal", "100000000"},
	{"gigapascal", "millibar", "10204081.632653"},
	{"gigapascal", "pound-force per square inch", "145037.743897"},
	{"gigapascal", "picopascal", "1000000000000000000000"},
	{"gigapascal", "centipascal", "100000000000"},
	{"gigapascal", "barye", "10000000000"},
	{"gigapascal", "millimeter of Water Column", "1019716212.977928"},
	{"gigapascal", "meter of Mercury", "75006.16827"},
	{"gigapascal", "petapascal", "0.000001"},
	{"gigapascal", "bar", "10204.081633"},
	{"gigapascal", "terapascal", "0.001"},
	{"gigapascal", "inch of Water Column", "4018646.519852"},
	{"gigapascal", "newton per square meter", "1000000000"},
	{"gigapascal", "millipascal", "1000000000000"},
	{"gigapascal", "exapascal", "0.000000001"},
	{"gigapascal", "attopascal", "999999999999999875848601600"},
	{"gigapascal", "meter of Water Column", "1019716.212978"},
	{"gigapascal", "kilopascal", "1000000"},
	{"gigapascal", "torr", "7500616.827045"},
	{"gigapascal", "micropascal", "1000000000000000"},
	{"gigapascal", "femtopascal", "999999999999999983222784"},
	{"gigapascal", "technical atmosphere", "10197.16213"},
	{"gigapascal", "hectopascal", "10000000"},
	{"gigapascal", "nanopascal", "1000000000000000000"},
	{"gigapascal", "millimeter of Mercury", "75006168.270447"},
	{"gigapascal", "standard atmosphere", "9869.205999"},
	{"gigapascal", "centibar", "1020408.163265"},
	{"gigapascal", "pascal", "1000000000"},
	{"gigapascal", "megapascal", "1000"},
	{"gigapascal", "microbar", "10204081632.653063"},
	{"gigapascal", "centimeter of Water Column", "101971621.297793"},
	{"gigapascal", "centimeter of Mercury", "7500616.827045"},
	{"gigapascal", "decipascal", "10000000000"},
	{"centipascal", "inch of Mercury", "0.000002952999"},
	{"centipascal", "decapascal", "0.001"},
	{"centipascal", "millibar", "0.0001020408"},
	{"centipascal", "pound-force per square inch", "0.000001450377"},
	{"centipascal", "picopascal", "10000000000"},
	{"centipascal", "gigapascal", "0.00000000001"},
	{"centipascal", "barye", "0.1"},
	{"centipascal", "millimeter of Water Column", "0.01019716"},
	{"centipascal", "meter of Mercury", "0.000000750062"},
	{"centipascal", "petapascal", "0.00000000000000001"},
	{"centipascal", "bar", "0.0000001020408"},
	{"centipascal", "terapascal", "0.00000000000001"},
	{"centipascal", "inch of Water Column", "0.0000401865"},
	{"centipascal", "newton per square meter", "0.01"},
	{"centipascal", "millipascal", "10"},
	{"centipascal", "exapascal", "0.00000000000000000001"},
	{"centipascal", "attopascal", "10000000000000000"},
	{"centipascal", "meter of Water Column", "0.00001019716"},
	{"centipascal", "kilopascal", "0.00001"},
	{"centipascal", "torr", "0.0000750062"},
	{"centipascal", "micropascal", "10000"},
	{"centipascal", "femtopascal", "10000000000000"},
	{"centipascal", "technical atmosphere", "0.0000001019716"},
	{"centipascal", "hectopascal", "0.0001"},
	{"centipascal", "nanopascal", "10000000"},
	{"centipascal", "millimeter of Mercury", "0.000750062"},
	{"centipascal", "standard atmosphere", "0.0000000986921"},
	{"centipascal", "centibar", "0.00001020408"},
	{"centipascal", "pascal", "0.01"},
	{"centipascal", "megapascal", "0.00000001"},
	{"centipascal", "microbar", "0.1020408"},
	{"centipascal", "centimeter of Water Column", "0.001019716"},
	{"centipascal", "centimeter of Mercury", "0.0000750062"},
	{"centipascal", "decipascal", "0.1"},
	{"barye", "inch of Mercury", "0.00002952999"},
	{"barye", "decapascal", "0.01"},
	{"barye", "millibar", "0.001020408"},
	{"barye", "pound-force per square inch", "0.00001450377"},
	{"barye", "picopascal", "100000000000"},
	{"barye", "gigapascal", "0.0000000001"},
	{"barye", "centipascal", "10"},
	{"barye", "millimeter of Water Column", "0.1019716"},
	{"barye", "meter of Mercury", "0.00000750062"},
	{"barye", "petapascal", "0.0000000000000001"},
	{"barye", "bar", "0.000001020408"},
	{"barye", "terapascal", "0.0000000000001"},
	{"barye", "inch of Water Column", "0.000401865"},
	{"barye", "newton per square meter", "0.1"},
	{"barye", "millipascal", "100"},
	{"barye", "exapascal", "0.0000000000000000001"},
	{"barye", "attopascal", "100000000000000000"},
	{"barye", "meter of Water Column", "0.0001019716"},
	{"barye", "kilopascal", "0.0001"},
	{"barye", "torr", "0.000750062"},
	{"barye", "micropascal", "100000"},
	{"barye", "femtopascal", "100000000000000"},
	{"barye", "technical atmosphere", "0.000001019716"},
	{"barye", "hectopascal", "0.001"},
	{"barye", "nanopascal", "100000000"},
	{"barye", "millimeter of Mercury", "0.00750062"},
	{"barye", "standard atmosphere", "0.000000986921"},
	{"barye", "centibar", "0.0001020408"},
	{"barye", "pascal", "0.1"},
	{"barye", "megapascal", "0.0000001"},
	{"barye", "microbar", "1.020408"},
	{"barye", "centimeter of Water Column", "0.01019716"},
	{"barye", "centimeter of Mercury", "0.000750062"},
	{"barye", "decipascal", "1"},
	{"millimeter of Water Column", "inch of Mercury", "0.0002895903"},
	{"millimeter of Water Column", "decapascal", "0.0980665"},
	{"millimeter of Water Column", "millibar", "0.01000679"},
	{"millimeter of Water Column", "pound-force per square inch", "0.0001422334"},
	{"millimeter of Water Column", "picopascal", "980665000000"},
	{"millimeter of Water Column", "gigapascal", "0.000000000980665"},
	{"millimeter of Water Column", "centipascal", "98.0665"},
	{"millimeter of Water Column", "barye", "9.80665"},
	{"millimeter of Water Column", "meter of Mercury", "0.0000735559"},
	{"millimeter of Water Column", "petapascal", "0.000000000000000980665"},
	{"millimeter of Water Column", "bar", "0.00001000679"},
	{"millimeter of Water Column", "terapascal", "0.000000000000980665"},
	{"millimeter of Water Column", "inch of Water Column", "0.00394095"},
	{"millimeter of Water Column", "newton per square meter", "0.980665"},
	{"millimeter of Water Column", "millipascal", "980.665"},
	{"millimeter of Water Column", "exapascal", "0.000000000000000000980665"},
	{"millimeter of Water Column", "attopascal", "980665000000000000"},
	{"millimeter of Water Column", "meter of Water Column", "0.001"},
	{"millimeter of Water Column", "kilopascal", "0.000980665"},
	{"millimeter of Water Column", "torr", "0.00735559"},
	{"millimeter of Water Column", "micropascal", "980665"},
	{"millimeter of Water Column", "femtopascal", "980664999999999.875"},
	{"millimeter of Water Column", "technical atmosphere", "0.00001"},
	{"millimeter of Water Column", "hectopascal", "0.00980665"},
	{"millimeter of Water Column", "nanopascal", "980665000"},
	{"millimeter of Water Column", "millimeter of Mercury", "0.0735559"},
	{"millimeter of Water Column", "standard atmosphere", "0.00000967838"},
	{"millimeter of Water Column", "centibar", "0.001000679"},
	{"millimeter of Water Column", "pascal", "0.980665"},
	{"millimeter of Water Column", "megapascal", "0.000000980665"},
	{"millimeter of Water Column", "microbar", "10.006786"},
	{"millimeter of Water Column", "centimeter of Water Column", "0.1"},
	{"millimeter of Water Column", "centimeter of Mercury", "0.00735559"},
	{"millimeter of Water Column", "decipascal", "9.80665"},
	{"meter of Mercury", "inch of Mercury", "3.937008"},
	{"meter of Mercury", "decapascal", "1333.223684"},
	{"meter of Mercury", "millibar", "136.043233"},
	{"meter of Mercury", "pound-force per square inch", "1.933678"},
	{"meter of Mercury", "picopascal", "13332236842100000"},
	{"meter of Mercury", "gigapascal", "0.00001333224"},
	{"meter of Mercury", "centipascal", "1333223.68421"},
	{"meter of Mercury", "barye", "133322.368421"},
	{"meter of Mercury", "millimeter of Water Column", "13595.098063"},
	{"meter of Mercury", "petapascal", "0.00000000001333224"},
	{"meter of Mercury", "bar", "0.1360432"},
	{"meter of Mercury", "terapascal", "0.00000001333224"},
	{"meter of Mercury", "inch of Water Column", "53.577547"},
	{"meter of Mercury", "newton per square meter", "13332.236842"},
	{"meter of Mercury", "millipascal", "13332236.8421"},
	{"meter of Mercury", "exapascal", "0.00000000000001333224"},
	{"meter of Mercury", "attopascal", "13332236842099999244288"},
	{"meter of Mercury", "meter of Water Column", "13.595098"},
	{"meter of Mercury", "kilopascal", "13.332237"},
	{"meter of Mercury", "torr", "100"},
	{"meter of Mercury", "micropascal", "13332236842.1"},
	{"meter of Mercury", "femtopascal", "13332236842099998720"},
	{"meter of Mercury", "technical atmosphere", "0.135951"},
	{"meter of Mercury", "hectopascal", "133.322368"},
	{"meter of Mercury", "nanopascal", "13332236842099.998047"},
	{"meter of Mercury", "millimeter of Mercury", "1000"},
	{"meter of Mercury", "standard atmosphere", "0.1315786"},
	{"meter of Mercury", "centibar", "13.604323"},
	{"meter of Mercury", "pascal", "13332.236842"},
	{"meter of Mercury", "megapascal", "0.01333224"},
	{"meter of Mercury", "microbar", "136043.233083"},
	{"meter of Mercury", "centimeter of Water Column", "1359.509806"},
	{"meter of Mercury", "centimeter of Mercury", "100"},
	{"meter of Mercury", "decipascal", "133322.368421"},
	{"petapascal", "inch of Mercury", "295299875080.794861"},
	{"petapascal", "decapascal", "100000000000000"},
	{"petapascal", "millibar", "10204081632653.060547"},
	{"petapascal", "pound-force per square inch", "145037743897.283112"},
	{"petapascal", "picopascal", "1000000000000000013287555072"},
	{"petapascal", "gigapascal", "1000000"},
	{"petapascal", "centipascal", "100000000000000000"},
	{"petapascal", "barye", "10000000000000000"},
	{"petapascal", "millimeter of Water Column", "1019716212977928.25"},
	{"petapascal", "meter of Mercury", "75006168270.446594"},
	{"petapascal", "bar", "10204081632.653061"},
	{"petapascal", "terapascal", "1000"},
	{"petapascal", "inch of Water Column", "4018646519852.11377"},
	{"petapascal", "newton per square meter", "1000000000000000"},
	{"petapascal", "millipascal", "1000000000000000000"},
	{"petapascal", "exapascal", "0.001"},
	{"petapascal", "attopascal", "999999999999999945575230987042816"},
	{"petapascal", "meter of Water Column", "1019716212977.928223"},
	{"petapascal", "kilopascal", "1000000000000"},
	{"petapascal", "torr", "7500616827044.65918"},
	{"petapascal", "micropascal", "1000000000000000000000"},
	{"petapascal", "femtopascal", "999999999999999879147136483328"},
	{"petapascal", "technical atmosphere", "10197162129.779282"},
	{"petapascal", "hectopascal", "10000000000000"},
	{"petapascal", "nanopascal", "999999999999999983222784"},
	{"petapascal", "millimeter of Mercury", "75006168270446.59375"},
	{"petapascal", "standard atmosphere", "9869205998.632101"},
	{"petapascal", "centibar", "1020408163265.30603"},
	{"petapascal", "pascal", "1000000000000000"},
	{"petapascal", "megapascal", "1000000000"},
	{"petapascal", "microbar", "10204081632653062"},
	{"petapascal", "centimeter of Water Column", "101971621297792.8125"},
	{"petapascal", "centimeter of Mercury", "7500616827044.65918"},
	{"petapascal", "decipascal", "10000000000000000"},
	{"bar", "inch of Mercury", "28.939388"},
	{"bar", "decapascal", "9800"},
	{"bar", "millibar", "1000"},
	{"bar", "pound-force per square inch", "14.213699"},
	{"bar", "picopascal", "98000000000000000"},
	{"bar", "gigapascal", "0.000098"},
	{"bar", "centipascal", "9800000"},
	{"bar", "barye", "980000"},
	{"bar", "millimeter of Water Column", "99932.188872"},
	{"bar", "meter of Mercury", "7.350604"},
	{"bar", "petapascal", "0.000000000098"},
	{"bar", "terapascal", "0.000000098"},
	{"bar", "inch of Water Column", "393.827359"},
	{"bar", "newton per square meter", "98000"},
	{"bar", "millipascal", "98000000"},
	{"bar", "exapascal", "0.000000000000098"},
	{"bar", "attopascal", "97999999999999995805696"},
	{"bar", "meter of Water Column", "99.932189"},
	{"bar", "kilopascal", "98"},
	{"bar", "torr", "735.060449"},
	{"bar", "micropascal", "98000000000"},
	{"bar", "femtopascal", "98000000000000000000"},
	{"bar", "technical atmosphere", "0.999322"},
	{"bar", "hectopascal", "980"},
	{"bar", "nanopascal", "98000000000000"},
	{"bar", "millimeter of Mercury", "7350.604491"},
	{"bar", "standard atmosphere", "0.967182"},
	{"bar", "centibar", "100"},
	{"bar", "pascal", "98000"},
	{"bar", "megapascal", "0.098"},
	{"bar", "microbar", "1000000"},
	{"bar", "centimeter of Water Column", "9993.218887"},
	{"bar", "centimeter of Mercury", "735.060449"},
	{"bar", "decipascal", "980000"},
	{"terapascal", "inch of Mercury", "295299875.080795"},
	{"terapascal", "decapascal", "100000000000"},
	{"terapascal", "millibar", "10204081632.653061"},
	{"terapascal", "pound-force per square inch", "145037743.897283"},
	{"terapascal", "picopascal", "999999999999999983222784"},
	{"terapascal", "gigapascal", "1000"},
	{"terapascal", "centipascal", "100000000000000"},
	{"terapascal", "barye", "10000000000000"},
	{"terapascal", "millimeter of Water Column", "1019716212977.928223"},
	{"terapascal", "meter of Mercury", "75006168.270447"},
	{"terapascal", "petapascal", "0.001"},
	{"terapascal", "bar", "10204081.632653"},
	{"terapascal", "inch of Water Column", "4018646519.852114"},
	{"terapascal", "newton per square meter", "1000000000000"},
	{"terapascal", "millipascal", "1000000000000000"},
	{"terapascal", "exapascal", "0.000001"},
	{"terapascal", "attopascal", "999999999999999879147136483328"},
	{"terapascal", "meter of Water Column", "1019716212.977928"},
	{"terapascal", "kilopascal", "1000000000"},
	{"terapascal", "torr", "7500616827.044659"},
	{"terapascal", "micropascal", "1000000000000000000"},
	{"terapascal", "femtopascal", "999999999999999875848601600"},
	{"terapascal", "technical atmosphere", "10197162.129779"},
	{"terapascal", "hectopascal", "10000000000"},
	{"terapascal", "nanopascal", "1000000000000000000000"},
	{"terapascal", "millimeter of Mercury", "75006168270.446579"},
	{"terapascal", "standard atmosphere", "9869205.998632"},
	{"terapascal", "centibar", "1020408163.265306"},
	{"terapascal", "pascal", "1000000000000"},
	{"terapascal", "megapascal", "1000000"},
	{"terapascal", "microbar", "10204081632653.0625"},
	{"terapascal", "centimeter of Water Column", "101971621297.792831"},
	{"terapascal", "centimeter of Mercury", "7500616827.044659"},
	{"terapascal", "decipascal", "10000000000000"},
	{"inch of Water Column", "inch of Mercury", "0.0734824"},
	{"inch of Water Column", "decapascal", "24.884"},
	{"inch of Water Column", "millibar", "2.539184"},
	{"inch of Water Column", "pound-force per square inch", "0.0360912"},
	{"inch of Water Column", "picopascal", "248840000000000"},
	{"inch of Water Column", "gigapascal", "0.00000024884"},
	{"inch of Water Column", "centipascal", "24884"},
	{"inch of Water Column", "barye", "2488.4"},
	{"inch of Water Column", "millimeter of Water Column", "253.746182"},
	{"inch of Water Column", "meter of Mercury", "0.01866453"},
	{"inch of Water Column", "petapascal", "0.00000000000024884"},
	{"inch of Water Column", "bar", "0.002539184"},
	{"inch of Water Column", "terapascal", "0.00000000024884"},
	{"inch of Water Column", "newton per square meter", "248.84"},
	{"inch of Water Column", "millipascal", "248840"},
	{"inch of Water Column", "exapascal", "0.00000000000000024884"},
	{"inch of Water Column", "attopascal", "248840000000000000000"},
	{"inch of Water Column", "meter of Water Column", "0.2537462"},
	{"inch of Water Column", "kilopascal", "0.24884"},
	{"inch of Water Column", "torr", "1.866453"},
	{"inch of Water Column", "micropascal", "248840000"},
	{"inch of Water Column", "femtopascal", "248840000000000000"},
	{"inch of Water Column", "technical atmosphere", "0.002537462"},
	{"inch of Water Column", "hectopascal", "2.4884"},
	{"inch of Water Column", "nanopascal", "248840000000"},
	{"inch of Water Column", "millimeter of Mercury", "18.664535"},
	{"inch of Water Column", "standard atmosphere", "0.002455853"},
	{"inch of Water Column", "centibar", "0.2539184"},
	{"inch of Water Column", "pascal", "248.84"},
	{"inch of Water Column", "megapascal", "0.00024884"},
	{"inch of Water Column", "microbar", "2539.183673"},
	{"inch of Water Column", "centimeter of Water Column", "25.374618"},
	{"inch of Water Column", "centimeter of Mercury", "1.866453"},
	{"inch of Water Column", "decipascal", "2488.4"},
	{"newton per square meter", "inch of Mercury", "0.0002952999"},
	{"newton per square meter", "decapascal", "0.1"},
	{"newton per square meter", "millibar", "0.01020408"},
	{"newton per square meter", "pound-force per square inch", "0.0001450377"},
	{"newton per square meter", "picopascal", "1000000000000"},
	{"newton per square meter", "gigapascal", "0.000000001"},
	{"newton per square meter", "centipascal", "100"},
	{"newton per square meter", "barye", "10"},
	{"newton per square meter", "millimeter of Water Column", "1.019716"},
	{"newton per square meter", "meter of Mercury", "0.0000750062"},
	{"newton per square meter", "petapascal", "0.000000000000001"},
	{"newton per square meter", "bar", "0.00001020408"},
	{"newton per square meter", "terapascal", "0.000000000001"},
	{"newton per square meter", "inch of Water Column", "0.00401865"},
	{"newton per square meter", "millipascal", "1000"},
	{"newton per square meter", "exapascal", "0.000000000000000001"},
	{"newton per square meter", "attopascal", "999999999999999872"},
	{"newton per square meter", "meter of Water Column", "0.001019716"},
	{"newton per square meter", "kilopascal", "0.001"},
	{"newton per square meter", "torr", "0.00750062"},
	{"newton per square meter", "micropascal", "1000000"},
	{"newton per square meter", "femtopascal", "999999999999999.875"},
	{"newton per square meter", "technical atmosphere", "0.00001019716"},
	{"newton per square meter", "hectopascal", "0.01"},
	{"newton per square meter", "nanopascal", "1000000000"},
	{"newton per square meter", "millimeter of Mercury", "0.0750062"},
	{"newton per square meter", "standard atmosphere", "0.00000986921"},
	{"newton per square meter", "centibar", "0.001020408"},
	{"newton per square meter", "pascal", "1"},
	{"newton per square meter", "megapascal", "0.000001"},
	{"newton per square meter", "microbar", "10.204082"},
	{"newton per square meter", "centimeter of Water Column", "0.1019716"},
	{"newton per square meter", "centimeter of Mercury", "0.00750062"},
	{"newton per square meter", "decipascal", "10"},
	{"millipascal", "inch of Mercury", "0.0000002952999"},
	{"millipascal", "decapascal", "0.0001"},
	{"millipascal", "millibar", "0.00001020408"},
	{"millipascal", "pound-force per square inch", "0.0000001450377"},
	{"millipascal", "picopascal", "1000000000"},
	{"millipascal", "gigapascal", "0.000000000001"},
	{"millipascal", "centipascal", "0.1"},
	{"millipascal", "barye", "0.01"},
	{"millipascal", "millimeter of Water Column", "0.001019716"},
	{"millipascal", "meter of Mercury", "0.0000000750062"},
	{"millipascal", "petapascal", "0.000000000000000001"},
	{"millipascal", "bar", "0.00000001020408"},
	{"millipascal", "terapascal", "0.000000000000001"},
	{"millipascal", "inch of Water Column", "0.00000401865"},
	{"millipascal", "newton per square meter", "0.001"},
	{"millipascal", "exapascal", "0.000000000000000000001"},
	{"millipascal", "attopascal", "1000000000000000"},
	{"millipascal", "meter of Water Column", "0.000001019716"},
	{"millipascal", "kilopascal", "0.000001"},
	{"millipascal", "torr", "0.00000750062"},
	{"millipascal", "micropascal", "1000"},
	{"millipascal", "femtopascal", "1000000000000"},
	{"millipascal", "technical atmosphere", "0.00000001019716"},
	{"millipascal", "hectopascal", "0.00001"},
	{"millipascal", "nanopascal", "1000000"},
	{"millipascal", "millimeter of Mercury", "0.0000750062"},
	{"millipascal", "standard atmosphere", "0.00000000986921"},
	{"millipascal", "centibar", "0.000001020408"},
	{"millipascal", "pascal", "0.001"},
	{"millipascal", "megapascal", "0.000000001"},
	{"millipascal", "microbar", "0.01020408"},
	{"millipascal", "centimeter of Water Column", "0.0001019716"},
	{"millipascal", "centimeter of Mercury", "0.00000750062"},
	{"millipascal", "decipascal", "0.01"},
	{"exapascal", "inch of Mercury", "295299875080794.875"},
	{"exapascal", "decapascal", "100000000000000000"},
	{"exapascal", "millibar", "10204081632653060"},
	{"exapascal", "pound-force per square inch", "145037743897283.125"},
	{"exapascal", "picopascal", "1000000000000000019884624838656"},
	{"exapascal", "gigapascal", "1000000000"},
	{"exapascal", "centipascal", "100000000000000000000"},
	{"exapascal", "barye", "10000000000000000000"},
	{"exapascal", "millimeter of Water Column", "1019716212977928192"},
	{"exapascal", "meter of Mercury", "75006168270446.59375"},
	{"exapascal", "petapascal", "1000"},
	{"exapascal", "bar", "10204081632653.060547"},
	{"exapascal", "terapascal", "1000000"},
	{"exapascal", "inch of Water Column", "4018646519852114"},
	{"exapascal", "newton per square meter", "1000000000000000000"},
	{"exapascal", "millipascal", "1000000000000000000000"},
	{"exapascal", "attopascal", "999999999999999894846684784341549056"},
	{"exapascal", "meter of Water Column", "1019716212977928.25"},
	{"exapascal", "kilopascal", "1000000000000000"},
	{"exapascal", "torr", "7500616827044659"},
	{"exapascal", "micropascal", "999999999999999983222784"},
	{"exapascal", "femtopascal", "999999999999999945575230987042816"},
	{"exapascal", "technical atmosphere", "10197162129779.283203"},
	{"exapascal", "hectopascal", "10000000000000000"},
	{"exapascal", "nanopascal", "999999999999999875848601600"},
	{"exapascal", "millimeter of Mercury", "75006168270446592"},
	{"exapascal", "standard atmosphere", "9869205998632.101562"},
	{"exapascal", "centibar", "1020408163265306"},
	{"exapascal", "pascal", "1000000000000000000"},
	{"exapascal", "megapascal", "1000000000000"},
	{"exapascal", "microbar", "10204081632653060096"},
	{"exapascal", "centimeter of Water Column", "101971621297792816"},
	{"exapascal", "centimeter of Mercury", "7500616827044659"},
	{"exapascal", "decipascal", "10000000000000000000"},
	{"attopascal", "inch of Mercury", "0.0000000000000000000002952999"},
	{"attopascal", "decapascal", "0.0000000000000000001"},
	{"attopascal", "millibar", "0.00000000000000000001020408"},
	{"attopascal", "pound-force per square inch", "0.0000000000000000000001450377"},
	{"attopascal", "picopascal", "0.000001"},
	{"attopascal", "gigapascal", "0.000000000000000000000000001"},
	{"attopascal", "centipascal", "0.0000000000000001"},
	{"attopascal", "barye", "0.00000000000000001"},
	{"attopascal", "millimeter of Water Column", "0.000000000000000001019716"},
	{"attopascal", "meter of Mercury", "0.0000000000000000000000750062"},
	{"attopascal", "petapascal", "0.000000000000000000000000000000001"},
	{"attopascal", "bar", "0.00000000000000000000001020408"},
	{"attopascal", "terapascal", "0.000000000000000000000000000001"},
	{"attopascal", "inch of Water Column", "0.00000000000000000000401865"},
	{"attopascal", "newton per square meter", "0.000000000000000001"},
	{"attopascal", "millipascal", "0.000000000000001"},
	{"attopascal", "exapascal", "0.000000000000000000000000000000000001"},
	{"attopascal", "meter of Water Column", "0.000000000000000000001019716"},
	{"attopascal", "kilopascal", "0.000000000000000000001"},
	{"attopascal", "torr", "0.00000000000000000000750062"},
	{"attopascal", "micropascal", "0.000000000001"},
	{"attopascal", "femtopascal", "0.001"},
	{"attopascal", "technical atmosphere", "0.00000000000000000000001019716"},
	{"attopascal", "hectopascal", "0.00000000000000000001"},
	{"attopascal", "nanopascal", "0.000000001"},
	{"attopascal", "millimeter of Mercury", "0.0000000000000000000750062"},
	{"attopascal", "standard atmosphere", "0.00000000000000000000000986921"},
	{"attopascal", "centibar", "0.000000000000000000001020408"},
	{"attopascal", "pascal", "0.000000000000000001"},
	{"attopascal", "megapascal", "0.000000000000000000000001"},
	{"attopascal", "microbar", "0.00000000000000001020408"},
	{"attopascal", "centimeter of Water Column", "0.0000000000000000001019716"},
	{"attopascal", "centimeter of Mercury", "0.00000000000000000000750062"},
	{"attopascal", "decipascal", "0.00000000000000001"},
	{"meter of Water Column", "inch of Mercury", "0.2895903"},
	{"meter of Water Column", "decapascal", "98.0665"},
	{"meter of Water Column", "millibar", "10.006786"},
	{"meter of Water Column", "pound-force per square inch", "0.1422334"},
	{"meter of Water Column", "picopascal", "980665000000000"},
	{"meter of Water Column", "gigapascal", "0.000000980665"},
	{"meter of Water Column", "centipascal", "98066.5"},
	{"meter of Water Column", "barye", "9806.65"},
	{"meter of Water Column", "millimeter of Water Column", "1000"},
	{"meter of Water Column", "meter of Mercury", "0.0735559"},
	{"meter of Water Column", "petapascal", "0.000000000000980665"},
	{"meter of Water Column", "bar", "0.01000679"},
	{"meter of Water Column", "terapascal", "0.000000000980665"},
	{"meter of Water Column", "inch of Water Column", "3.940946"},
	{"meter of Water Column", "newton per square meter", "980.665"},
	{"meter of Water Column", "millipascal", "980665"},
	{"meter of Water Column", "exapascal", "0.000000000000000980665"},
	{"meter of Water Column", "attopascal", "980664999999999836160"},
	{"meter of Water Column", "kilopascal", "0.980665"},
	{"meter of Water Column", "torr", "7.355592"},
	{"meter of Water Column", "micropascal", "980665000"},
	{"meter of Water Column", "femtopascal", "980664999999999872"},
	{"meter of Water Column", "technical atmosphere", "0.01"},
	{"meter of Water Column", "hectopascal", "9.80665"},
	{"meter of Water Column", "nanopascal", "980664999999.999878"},
	{"meter of Water Column", "millimeter of Mercury", "73.555924"},
	{"meter of Water Column", "standard atmosphere", "0.00967838"},
	{"meter of Water Column", "centibar", "1.000679"},
	{"meter of Water Column", "pascal", "980.665"},
	{"meter of Water Column", "megapascal", "0.000980665"},
	{"meter of Water Column", "microbar", "10006.785714"},
	{"meter of Water Column", "centimeter of Water Column", "100"},
	{"meter of Water Column", "centimeter of Mercury", "7.355592"},
	{"meter of Water Column", "decipascal", "9806.65"},
	{"kilopascal", "inch of Mercury", "0.2952999"},
	{"kilopascal", "decapascal", "100"},
	{"kilopascal", "millibar", "10.204082"},
	{"kilopascal", "pound-force per square inch", "0.1450377"},
	{"kilopascal", "picopascal", "1000000000000000"},
	{"kilopascal", "gigapascal", "0.000001"},
	{"kilopascal", "centipascal", "100000"},
	{"kilopascal", "barye", "10000"},
	{"kilopascal", "millimeter of Water Column", "1019.716213"},
	{"kilopascal", "meter of Mercury", "0.0750062"},
	{"kilopascal", "petapascal", "0.000000000001"},
	{"kilopascal", "bar", "0.01020408"},
	{"kilopascal", "terapascal", "0.000000001"},
	{"kilopascal", "inch of Water Column", "4.018647"},
	{"kilopascal", "newton per square meter", "1000"},
	{"kilopascal", "millipascal", "1000000"},
	{"kilopascal", "exapascal", "0.000000000000001"},
	{"kilopascal", "attopascal", "999999999999999868928"},
	{"kilopascal", "meter of Water Column", "1.019716"},
	{"kilopascal", "torr", "7.500617"},
	{"kilopascal", "micropascal", "1000000000"},
	{"kilopascal", "femtopascal", "999999999999999872"},
	{"kilopascal", "technical atmosphere", "0.01019716"},
	{"kilopascal", "hectopascal", "10"},
	{"kilopascal", "nanopascal", "999999999999.999878"},
	{"kilopascal", "millimeter of Mercury", "75.006168"},
	{"kilopascal", "standard atmosphere", "0.00986921"},
	{"kilopascal", "centibar", "1.020408"},
	{"kilopascal", "pascal", "1000"},
	{"kilopascal", "megapascal", "0.001"},
	{"kilopascal", "microbar", "10204.081633"},
	{"kilopascal", "centimeter of Water Column", "101.971621"},
	{"kilopascal", "centimeter of Mercury", "7.500617"},
	{"kilopascal", "decipascal", "10000"},
	{"torr", "inch of Mercury", "0.0393701"},
	{"torr", "decapascal", "13.332237"},
	{"torr", "millibar", "1.360432"},
	{"torr", "pound-force per square inch", "0.01933678"},
	{"torr", "picopascal", "133322368421000"},
	{"torr", "gigapascal", "0.0000001333224"},
	{"torr", "centipascal", "13332.236842"},
	{"torr", "barye", "1333.223684"},
	{"torr", "millimeter of Water Column", "135.950981"},
	{"torr", "meter of Mercury", "0.01"},
	{"torr", "petapascal", "0.0000000000001333224"},
	{"torr", "bar", "0.001360432"},
	{"torr", "terapascal", "0.0000000001333224"},
	{"torr", "inch of Water Column", "0.535775"},
	{"torr", "newton per square meter", "133.322368"},
	{"torr", "millipascal", "133322.368421"},
	{"torr", "exapascal", "0.0000000000000001333224"},
	{"torr", "attopascal", "133322368420999987200"},
	{"torr", "meter of Water Column", "0.135951"},
	{"torr", "kilopascal", "0.1333224"},
	{"torr", "micropascal", "133322368.421"},
	{"torr", "femtopascal", "133322368420999984"},
	{"torr", "technical atmosphere", "0.00135951"},
	{"torr", "hectopascal", "1.333224"},
	{"torr", "nanopascal", "133322368420.999985"},
	{"torr", "millimeter of Mercury", "10"},
	{"torr", "standard atmosphere", "0.001315786"},
	{"torr", "centibar", "0.1360432"},
	{"torr", "pascal", "133.322368"},
	{"torr", "megapascal", "0.0001333224"},
	{"torr", "microbar", "1360.432331"},
	{"torr", "centimeter of Water Column", "13.595098"},
	{"torr", "centimeter of Mercury", "1"},
	{"torr", "decipascal", "1333.223684"},
	{"micropascal", "inch of Mercury", "0.0000000002952999"},
	{"micropascal", "decapascal", "0.0000001"},
	{"micropascal", "millibar", "0.00000001020408"},
	{"micropascal", "pound-force per square inch", "0.0000000001450377"},
	{"micropascal", "picopascal", "1000000"},
	{"micropascal", "gigapascal", "0.000000000000001"},
	{"micropascal", "centipascal", "0.0001"},
	{"micropascal", "barye", "0.00001"},
	{"micropascal", "millimeter of Water Column", "0.000001019716"},
	{"micropascal", "meter of Mercury", "0.0000000000750062"},
	{"micropascal", "petapascal", "0.000000000000000000001"},
	{"micropascal", "bar", "0.00000000001020408"},
	{"micropascal", "terapascal", "0.000000000000000001"},
	{"micropascal", "inch of Water Column", "0.00000000401865"},
	{"micropascal", "newton per square meter", "0.000001"},
	{"micropascal", "millipascal", "0.001"},
	{"micropascal", "exapascal", "0.000000000000000000000001"},
	{"micropascal", "attopascal", "999999999999.999878"},
	{"micropascal", "meter of Water Column", "0.000000001019716"},
	{"micropascal", "kilopascal", "0.000000001"},
	{"micropascal", "torr", "0.00000000750062"},
	{"micropascal", "femtopascal", "1000000000"},
	{"micropascal", "technical atmosphere", "0.00000000001019716"},
	{"micropascal", "hectopascal", "0.00000001"},
	{"micropascal", "nanopascal", "1000"},
	{"micropascal", "millimeter of Mercury", "0.0000000750062"},
	{"micropascal", "standard atmosphere", "0.00000000000986921"},
	{"micropascal", "centibar", "0.000000001020408"},
	{"micropascal", "pascal", "0.000001"},
	{"micropascal", "megapascal", "0.000000000001"},
	{"micropascal", "microbar", "0.00001020408"},
	{"micropascal", "centimeter of Water Column", "0.0000001019716"},
	{"micropascal", "centimeter of Mercury", "0.00000000750062"},
	{"micropascal", "decipascal", "0.00001"},
	{"femtopascal", "inch of Mercury", "0.0000000000000000002952999"},
	{"femtopascal", "decapascal", "0.0000000000000001"},
	{"femtopascal", "millibar", "0.00000000000000001020408"},
	{"femtopascal", "pound-force per square inch", "0.0000000000000000001450377"},
	{"femtopascal", "picopascal", "0.001"},
	{"femtopascal", "gigapascal", "0.000000000000000000000001"},
	{"femtopascal", "centipascal", "0.0000000000001"},
	{"femtopascal", "barye", "0.00000000000001"},
	{"femtopascal", "millimeter of Water Column", "0.000000000000001019716"},
	{"femtopascal", "meter of Mercury", "0.0000000000000000000750062"},
	{"femtopascal", "petapascal", "0.000000000000000000000000000001"},
	{"femtopascal", "bar", "0.00000000000000000001020408"},
	{"femtopascal", "terapascal", "0.000000000000000000000000001"},
	{"femtopascal", "inch of Water Column", "0.00000000000000000401865"},
	{"femtopascal", "newton per square meter", "0.000000000000001"},
	{"femtopascal", "millipascal", "0.000000000001"},
	{"femtopascal", "exapascal", "0.000000000000000000000000000000001"},
	{"femtopascal", "attopascal", "1000"},
	{"femtopascal", "meter of Water Column", "0.000000000000000001019716"},
	{"femtopascal", "kilopascal", "0.000000000000000001"},
	{"femtopascal", "torr", "0.00000000000000000750062"},
	{"femtopascal", "micropascal", "0.000000001"},
	{"femtopascal", "technical atmosphere", "0.00000000000000000001019716"},
	{"femtopascal", "hectopascal", "0.00000000000000001"},
	{"femtopascal", "nanopascal", "0.000001"},
	{"femtopascal", "millimeter of Mercury", "0.0000000000000000750062"},
	{"femtopascal", "standard atmosphere", "0.00000000000000000000986921"},
	{"femtopascal", "centibar", "0.000000000000000001020408"},
	{"femtopascal", "pascal", "0.000000000000001"},
	{"femtopascal", "megapascal", "0.000000000000000000001"},
	{"femtopascal", "microbar", "0.00000000000001020408"},
	{"femtopascal", "centimeter of Water Column", "0.0000000000000001019716"},
	{"femtopascal", "centimeter of Mercury", "0.00000000000000000750062"},
	{"femtopascal", "decipascal", "0.00000000000001"},
	{"technical atmosphere", "inch of Mercury", "28.959025"},
	{"technical atmosphere", "decapascal", "9806.65"},
	{"technical atmosphere", "millibar", "1000.678571"},
	{"technical atmosphere", "pound-force per square inch", "14.223344"},
	{"technical atmosphere", "picopascal", "98066500000000000"},
	{"technical atmosphere", "gigapascal", "0.0000980665"},
	{"technical atmosphere", "centipascal", "9806650"},
	{"technical atmosphere", "barye", "980665"},
	{"technical atmosphere", "millimeter of Water Column", "100000"},
	{"technical atmosphere", "meter of Mercury", "7.355592"},
	{"technical atmosphere", "petapascal", "0.0000000000980665"},
	{"technical atmosphere", "bar", "1.000679"},
	{"technical atmosphere", "terapascal", "0.0000000980665"},
	{"technical atmosphere", "inch of Water Column", "394.094599"},
	{"technical atmosphere", "newton per square meter", "98066.5"},
	{"technical atmosphere", "millipascal", "98066500"},
	{"technical atmosphere", "exapascal", "0.0000000000000980665"},
	{"technical atmosphere", "attopascal", "98066499999999987810304"},
	{"technical atmosphere", "meter of Water Column", "100"},
	{"technical atmosphere", "kilopascal", "98.0665"},
	{"technical atmosphere", "torr", "735.55924"},
	{"technical atmosphere", "micropascal", "98066500000"},
	{"technical atmosphere", "femtopascal", "98066500000000000000"},
	{"technical atmosphere", "hectopascal", "980.665"},
	{"technical atmosphere", "nanopascal", "98066500000000"},
	{"technical atmosphere", "millimeter of Mercury", "7355.592401"},
	{"technical atmosphere", "standard atmosphere", "0.967838"},
	{"technical atmosphere", "centibar", "100.067857"},
	{"technical atmosphere", "pascal", "98066.5"},
	{"technical atmosphere", "megapascal", "0.0980665"},
	{"technical atmosphere", "microbar", "1000678.571429"},
	{"technical atmosphere", "centimeter of Water Column", "10000"},
	{"technical atmosphere", "centimeter of Mercury", "735.55924"},
	{"technical atmosphere", "decipascal", "980665"},
	{"hectopascal", "inch of Mercury", "0.02952999"},
	{"hectopascal", "decapascal", "10"},
	{"hectopascal", "millibar", "1.020408"},
	{"hectopascal", "pound-force per square inch", "0.01450377"},
	{"hectopascal", "picopascal", "100000000000000"},
	{"hectopascal", "gigapascal", "0.0000001"},
	{"hectopascal", "centipascal", "10000"},
	{"hectopascal", "barye", "1000"},
	{"hectopascal", "millimeter of Water Column", "101.971621"},
	{"hectopascal", "meter of Mercury", "0.00750062"},
	{"hectopascal", "petapascal", "0.0000000000001"},
	{"hectopascal", "bar", "0.001020408"},
	{"hectopascal", "terapascal", "0.0000000001"},
	{"hectopascal", "inch of Water Column", "0.401865"},
	{"hectopascal", "newton per square meter", "100"},
	{"hectopascal", "millipascal", "100000"},
	{"hectopascal", "exapascal", "0.0000000000000001"},
	{"hectopascal", "attopascal", "100000000000000000000"},
	{"hectopascal", "meter of Water Column", "0.1019716"},
	{"hectopascal", "kilopascal", "0.1"},
	{"hectopascal", "torr", "0.750062"},
	{"hectopascal", "micropascal", "100000000"},
	{"hectopascal", "femtopascal", "100000000000000000"},
	{"hectopascal", "technical atmosphere", "0.001019716"},
	{"hectopascal", "nanopascal", "100000000000"},
	{"hectopascal", "millimeter of Mercury", "7.500617"},
	{"hectopascal", "standard atmosphere", "0.000986921"},
	{"hectopascal", "centibar", "0.1020408"},
	{"hectopascal", "pascal", "100"},
	{"hectopascal", "megapascal", "0.0001"},
	{"hectopascal", "microbar", "1020.408163"},
	{"hectopascal", "centimeter of Water Column", "10.197162"},
	{"hectopascal", "centimeter of Mercury", "0.750062"},
	{"hectopascal", "decipascal", "1000"},
	{"nanopascal", "inch of Mercury", "0.0000000000002952999"},
	{"nanopascal", "decapascal", "0.0000000001"},
	{"nanopascal", "millibar", "0.00000000001020408"},
	{"nanopascal", "pound-force per square inch", "0.0000000000001450377"},
	{"nanopascal", "picopascal", "1000"},
	{"nanopascal", "gigapascal", "0.000000000000000001"},
	{"nanopascal", "centipascal", "0.0000001"},
	{"nanopascal", "barye", "0.00000001"},
	{"nanopascal", "millimeter of Water Column", "0.000000001019716"},
	{"nanopascal", "meter of Mercury", "0.0000000000000750062"},
	{"nanopascal", "petapascal", "0.000000000000000000000001"},
	{"nanopascal", "bar", "0.00000000000001020408"},
	{"nanopascal", "terapascal", "0.000000000000000000001"},
	{"nanopascal", "inch of Water Column", "0.00000000000401865"},
	{"nanopascal", "newton per square meter", "0.000000001"},
	{"nanopascal", "millipascal", "0.000001"},
	{"nanopascal", "exapascal", "0.000000000000000000000000001"},
	{"nanopascal", "attopascal", "1000000000"},
	{"nanopascal", "meter of Water Column", "0.000000000001019716"},
	{"nanopascal", "kilopascal", "0.000000000001"},
	{"nanopascal", "torr", "0.00000000000750062"},
	{"nanopascal", "micropascal", "0.001"},
	{"nanopascal", "femtopascal", "1000000"},
	{"nanopascal", "technical atmosphere", "0.00000000000001019716"},
	{"nanopascal", "hectopascal", "0.00000000001"},
	{"nanopascal", "millimeter of Mercury", "0.0000000000750062"},
	{"nanopascal", "standard atmosphere", "0.00000000000000986921"},
	{"nanopascal", "centibar", "0.000000000001020408"},
	{"nanopascal", "pascal", "0.000000001"},
	{"nanopascal", "megapascal", "0.000000000000001"},
	{"nanopascal", "microbar", "0.00000001020408"},
	{"nanopascal", "centimeter of Water Column", "0.0000000001019716"},
	{"nanopascal", "centimeter of Mercury", "0.00000000000750062"},
	{"nanopascal", "decipascal", "0.00000001"},
	{"millimeter of Mercury", "inch of Mercury", "0.00393701"},
	{"millimeter of Mercury", "decapascal", "1.333224"},
	{"millimeter of Mercury", "millibar", "0.1360432"},
	{"millimeter of Mercury", "pound-force per square inch", "0.001933678"},
	{"millimeter of Mercury", "picopascal", "13332236842099.998047"},
	{"millimeter of Mercury", "gigapascal", "0.00000001333224"},
	{"millimeter of Mercury", "centipascal", "1333.223684"},
	{"millimeter of Mercury", "barye", "133.322368"},
	{"millimeter of Mercury", "millimeter of Water Column", "13.595098"},
	{"millimeter of Mercury", "meter of Mercury", "0.001"},
	{"millimeter of Mercury", "petapascal", "0.00000000000001333224"},
	{"millimeter of Mercury", "bar", "0.0001360432"},
	{"millimeter of Mercury", "terapascal", "0.00000000001333224"},
	{"millimeter of Mercury", "inch of Water Column", "0.0535775"},
	{"millimeter of Mercury", "newton per square meter", "13.332237"},
	{"millimeter of Mercury", "millipascal", "13332.236842"},
	{"millimeter of Mercury", "exapascal", "0.00000000000000001333224"},
	{"millimeter of Mercury", "attopascal", "13332236842099996672"},
	{"millimeter of Mercury", "meter of Water Column", "0.0135951"},
	{"millimeter of Mercury", "kilopascal", "0.01333224"},
	{"millimeter of Mercury", "torr", "0.1"},
	{"millimeter of Mercury", "micropascal", "13332236.8421"},
	{"millimeter of Mercury", "femtopascal", "13332236842099998"},
	{"millimeter of Mercury", "technical atmosphere", "0.000135951"},
	{"millimeter of Mercury", "hectopascal", "0.1333224"},
	{"millimeter of Mercury", "nanopascal", "13332236842.099998"},
	{"millimeter of Mercury", "standard atmosphere", "0.0001315786"},
	{"millimeter of Mercury", "centibar", "0.01360432"},
	{"millimeter of Mercury", "pascal", "13.332237"},
	{"millimeter of Mercury", "megapascal", "0.00001333224"},
	{"millimeter of Mercury", "microbar", "136.043233"},
	{"millimeter of Mercury", "centimeter of Water Column", "1.35951"},
	{"millimeter of Mercury", "centimeter of Mercury", "0.1"},
	{"millimeter of Mercury", "decipascal", "133.322368"},
	{"standard atmosphere", "inch of Mercury", "29.921341"},
	{"standard atmosphere", "decapascal", "10132.52738"},
	{"standard atmosphere", "millibar", "1033.931365"},
	{"standard atmosphere", "pound-force per square inch", "14.695989"},
	{"standard atmosphere", "picopascal", "101325273800000000"},
	{"standard atmosphere", "gigapascal", "0.0001013253"},
	{"standard atmosphere", "centipascal", "10132527.38"},
	{"standard atmosphere", "barye", "1013252.738"},
	{"standard atmosphere", "millimeter of Water Column", "103323.024478"},
	{"standard atmosphere", "meter of Mercury", "7.600021"},
	{"standard atmosphere", "petapascal", "0.0000000001013253"},
	{"standard atmosphere", "bar", "1.033931"},
	{"standard atmosphere", "terapascal", "0.0000001013253"},
	{"standard atmosphere", "inch of Water Column", "407.190459"},
	{"standard atmosphere", "newton per square meter", "101325.2738"},
	{"standard atmosphere", "millipascal", "101325273.8"},
	{"standard atmosphere", "exapascal", "0.0000000000001013253"},
	{"standard atmosphere", "attopascal", "101325273799999990267904"},
	{"standard atmosphere", "meter of Water Column", "103.323024"},
	{"standard atmosphere", "kilopascal", "101.325274"},
	{"standard atmosphere", "torr", "760.002054"},
	{"standard atmosphere", "micropascal", "101325273800"},
	{"standard atmosphere", "femtopascal", "101325273799999979520"},
	{"standard atmosphere", "technical atmosphere", "1.03323"},
	{"standard atmosphere", "hectopascal", "1013.252738"},
	{"standard atmosphere", "nanopascal", "101325273799999.984375"},
	{"standard atmosphere", "millimeter of Mercury", "7600.020537"},
	{"standard atmosphere", "centibar", "103.393137"},
	{"standard atmosphere", "pascal", "101325.2738"},
	{"standard atmosphere", "megapascal", "0.1013253"},
	{"standard atmosphere", "microbar", "1033931.365306"},
	{"standard atmosphere", "centimeter of Water Column", "10332.302448"},
	{"standard atmosphere", "centimeter of Mercury", "760.002054"},
	{"standard atmosphere", "decipascal", "1013252.738"},
	{"centibar", "inch of Mercury", "0.2893939"},
	{"centibar", "decapascal", "98"},
	{"centibar", "millibar", "10"},
	{"centibar", "pound-force per square inch", "0.142137"},
	{"centibar", "picopascal", "980000000000000"},
	{"centibar", "gigapascal", "0.00000098"},
	{"centibar", "centipascal", "98000"},
	{"centibar", "barye", "9800"},
	{"centibar", "millimeter of Water Column", "999.321889"},
	{"centibar", "meter of Mercury", "0.073506"},
	{"centibar", "petapascal", "0.00000000000098"},
	{"centibar", "bar", "0.01"},
	{"centibar", "terapascal", "0.00000000098"},
	{"centibar", "inch of Water Column", "3.938274"},
	{"centibar", "newton per square meter", "980"},
	{"centibar", "millipascal", "980000"},
	{"centibar", "exapascal", "0.00000000000000098"},
	{"centibar", "attopascal", "979999999999999868928"},
	{"centibar", "meter of Water Column", "0.999322"},
	{"centibar", "kilopascal", "0.98"},
	{"centibar", "torr", "7.350604"},
	{"centibar", "micropascal", "980000000"},
	{"centibar", "femtopascal", "979999999999999872"},
	{"centibar", "technical atmosphere", "0.00999322"},
	{"centibar", "hectopascal", "9.8"},
	{"centibar", "nanopascal", "979999999999.999878"},
	{"centibar", "millimeter of Mercury", "73.506045"},
	{"centibar", "standard atmosphere", "0.00967182"},
	{"centibar", "pascal", "980"},
	{"centibar", "megapascal", "0.00098"},
	{"centibar", "microbar", "10000"},
	{"centibar", "centimeter of Water Column", "99.932189"},
	{"centibar", "centimeter of Mercury", "7.350604"},
	{"centibar", "decipascal", "9800"},
	{"pascal", "inch of Mercury", "0.0002952999"},
	{"pascal", "decapascal", "0.1"},
	{"pascal", "millibar", "0.01020408"},
	{"pascal", "pound-force per square inch", "0.0001450377"},
	{"pascal", "picopascal", "1000000000000"},
	{"pascal", "gigapascal", "0.000000001"},
	{"pascal", "centipascal", "100"},
	{"pascal", "barye", "10"},
	{"pascal", "millimeter of Water Column", "1.019716"},
	{"pascal", "meter of Mercury", "0.0000750062"},
	{"pascal", "petapascal", "0.000000000000001"},
	{"pascal", "bar", "0.00001020408"},
	{"pascal", "terapascal", "0.000000000001"},
	{"pascal", "inch of Water Column", "0.00401865"},
	{"pascal", "newton per square meter", "1"},
	{"pascal", "millipascal", "1000"},
	{"pascal", "exapascal", "0.000000000000000001"},
	{"pascal", "attopascal", "999999999999999872"},
	{"pascal", "meter of Water Column", "0.001019716"},
	{"pascal", "kilopascal", "0.001"},
	{"pascal", "torr", "0.00750062"},
	{"pascal", "micropascal", "1000000"},
	{"pascal", "femtopascal", "999999999999999.875"},
	{"pascal", "technical atmosphere", "0.00001019716"},
	{"pascal", "hectopascal", "0.01"},
	{"pascal", "nanopascal", "1000000000"},
	{"pascal", "millimeter of Mercury", "0.0750062"},
	{"pascal", "standard atmosphere", "0.00000986921"},
	{"pascal", "centibar", "0.001020408"},
	{"pascal", "megapascal", "0.000001"},
	{"pascal", "microbar", "10.204082"},
	{"pascal", "centimeter of Water Column", "0.1019716"},
	{"pascal", "centimeter of Mercury", "0.00750062"},
	{"pascal", "decipascal", "10"},
	{"megapascal", "inch of Mercury", "295.299875"},
	{"megapascal", "decapascal", "100000"},
	{"megapascal", "millibar", "10204.081633"},
	{"megapascal", "pound-force per square inch", "145.037744"},
	{"megapascal", "picopascal", "1000000000000000000"},
	{"megapascal", "gigapascal", "0.001"},
	{"megapascal", "centipascal", "100000000"},
	{"megapascal", "barye", "10000000"},
	{"megapascal", "millimeter of Water Column", "1019716.212978"},
	{"megapascal", "meter of Mercury", "75.006168"},
	{"megapascal", "petapascal", "0.000000001"},
	{"megapascal", "bar", "10.204082"},
	{"megapascal", "terapascal", "0.000001"},
	{"megapascal", "inch of Water Column", "4018.64652"},
	{"megapascal", "newton per square meter", "1000000"},
	{"megapascal", "millipascal", "1000000000"},
	{"megapascal", "exapascal", "0.000000000001"},
	{"megapascal", "attopascal", "999999999999999983222784"},
	{"megapascal", "meter of Water Column", "1019.716213"},
	{"megapascal", "kilopascal", "1000"},
	{"megapascal", "torr", "7500.616827"},
	{"megapascal", "micropascal", "1000000000000"},
	{"megapascal", "femtopascal", "999999999999999868928"},
	{"megapascal", "technical atmosphere", "10.197162"},
	{"megapascal", "hectopascal", "10000"},
	{"megapascal", "nanopascal", "1000000000000000"},
	{"megapascal", "millimeter of Mercury", "75006.16827"},
	{"megapascal", "standard atmosphere", "9.869206"},
	{"megapascal", "centibar", "1020.408163"},
	{"megapascal", "pascal", "1000000"},
	{"megapascal", "microbar", "10204081.632653"},
	{"megapascal", "centimeter of Water Column", "101971.621298"},
	{"megapascal", "centimeter of Mercury", "7500.616827"},
	{"megapascal", "decipascal", "10000000"},
	{"microbar", "inch of Mercury", "0.00002893939"},
	{"microbar", "decapascal", "0.0098"},
	{"microbar", "millibar", "0.001"},
	{"microbar", "pound-force per square inch", "0.0000142137"},
	{"microbar", "picopascal", "97999999999.999985"},
	{"microbar", "gigapascal", "0.000000000098"},
	{"microbar", "centipascal", "9.8"},
	{"microbar", "barye", "0.98"},
	{"microbar", "millimeter of Water Column", "0.0999322"},
	{"microbar", "meter of Mercury", "0.0000073506"},
	{"microbar", "petapascal", "0.000000000000000098"},
	{"microbar", "bar", "0.000001"},
	{"microbar", "terapascal", "0.000000000000098"},
	{"microbar", "inch of Water Column", "0.000393827"},
	{"microbar", "newton per square meter", "0.098"},
	{"microbar", "millipascal", "98"},
	{"microbar", "exapascal", "0.000000000000000000098"},
	{"microbar", "attopascal", "97999999999999984"},
	{"microbar", "meter of Water Column", "0.0000999322"},
	{"microbar", "kilopascal", "0.000098"},
	{"microbar", "torr", "0.00073506"},
	{"microbar", "micropascal", "98000"},
	{"microbar", "femtopascal", "97999999999999.984375"},
	{"microbar", "technical atmosphere", "0.000000999322"},
	{"microbar", "hectopascal", "0.00098"},
	{"microbar", "nanopascal", "98000000"},
	{"microbar", "millimeter of Mercury", "0.0073506"},
	{"microbar", "standard atmosphere", "0.000000967182"},
	{"microbar", "centibar", "0.0001"},
	{"microbar", "pascal", "0.098"},
	{"microbar", "megapascal", "0.000000098"},
	{"microbar", "centimeter of Water Column", "0.00999322"},
	{"microbar", "centimeter of Mercury", "0.00073506"},
	{"microbar", "decipascal", "0.98"},
	{"centimeter of Water Column", "inch of Mercury", "0.002895903"},
	{"centimeter of Water Column", "decapascal", "0.980665"},
	{"centimeter of Water Column", "millibar", "0.1000679"},
	{"centimeter of Water Column", "pound-force per square inch", "0.001422334"},
	{"centimeter of Water Column", "picopascal", "9806650000000"},
	{"centimeter of Water Column", "gigapascal", "0.00000000980665"},
	{"centimeter of Water Column", "centipascal", "980.665"},
	{"centimeter of Water Column", "barye", "98.0665"},
	{"centimeter of Water Column", "millimeter of Water Column", "10"},
	{"centimeter of Water Column", "meter of Mercury", "0.000735559"},
	{"centimeter of Water Column", "petapascal", "0.00000000000000980665"},
	{"centimeter of Water Column", "bar", "0.0001000679"},
	{"centimeter of Water Column", "terapascal", "0.00000000000980665"},
	{"centimeter of Water Column", "inch of Water Column", "0.0394095"},
	{"centimeter of Water Column", "newton per square meter", "9.80665"},
	{"centimeter of Water Column", "millipascal", "9806.65"},
	{"centimeter of Water Column", "exapascal", "0.00000000000000000980665"},
	{"centimeter of Water Column", "attopascal", "9806649999999997952"},
	{"centimeter of Water Column", "meter of Water Column", "0.01"},
	{"centimeter of Water Column", "kilopascal", "0.00980665"},
	{"centimeter of Water Column", "torr", "0.0735559"},
	{"centimeter of Water Column", "micropascal", "9806650"},
	{"centimeter of Water Column", "femtopascal", "9806649999999998"},
	{"centimeter of Water Column", "technical atmosphere", "0.0001"},
	{"centimeter of Water Column", "hectopascal", "0.0980665"},
	{"centimeter of Water Column", "nanopascal", "9806649999.999998"},
	{"centimeter of Water Column", "millimeter of Mercury", "0.735559"},
	{"centimeter of Water Column", "standard atmosphere", "0.0000967838"},
	{"centimeter of Water Column", "centibar", "0.01000679"},
	{"centimeter of Water Column", "pascal", "9.80665"},
	{"centimeter of Water Column", "megapascal", "0.00000980665"},
	{"centimeter of Water Column", "microbar", "100.067857"},
	{"centimeter of Water Column", "centimeter of Mercury", "0.0735559"},
	{"centimeter of Water Column", "decipascal", "98.0665"},
	{"centimeter of Mercury", "inch of Mercury", "0.0393701"},
	{"centimeter of Mercury", "decapascal", "13.332237"},
	{"centimeter of Mercury", "millibar", "1.360432"},
	{"centimeter of Mercury", "pound-force per square inch", "0.01933678"},
	{"centimeter of Mercury", "picopascal", "133322368421000"},
	{"centimeter of Mercury", "gigapascal", "0.0000001333224"},
	{"centimeter of Mercury", "centipascal", "13332.236842"},
	{"centimeter of Mercury", "barye", "1333.223684"},
	{"centimeter of Mercury", "millimeter of Water Column", "135.950981"},
	{"centimeter of Mercury", "meter of Mercury", "0.01"},
	{"centimeter of Mercury", "petapascal", "0.0000000000001333224"},
	{"centimeter of Mercury", "bar", "0.001360432"},
	{"centimeter of Mercury", "terapascal", "0.0000000001333224"},
	{"centimeter of Mercury", "inch of Water Column", "0.535775"},
	{"centimeter of Mercury", "newton per square meter", "133.322368"},
	{"centimeter of Mercury", "millipascal", "133322.368421"},
	{"centimeter of Mercury", "exapascal", "0.0000000000000001333224"},
	{"centimeter of Mercury", "attopascal", "133322368420999987200"},
	{"centimeter of Mercury", "meter of Water Column", "0.135951"},
	{"centimeter of Mercury", "kilopascal", "0.1333224"},
	{"centimeter of Mercury", "torr", "1"},
	{"centimeter of Mercury", "micropascal", "133322368.421"},
	{"centimeter of Mercury", "femtopascal", "133322368420999984"},
	{"centimeter of Mercury", "technical atmosphere", "0.00135951"},
	{"centimeter of Mercury", "hectopascal", "1.333224"},
	{"centimeter of Mercury", "nanopascal", "133322368420.999985"},
	{"centimeter of Mercury", "millimeter of Mercury", "10"},
	{"centimeter of Mercury", "standard atmosphere", "0.001315786"},
	{"centimeter of Mercury", "centibar", "0.1360432"},
	{"centimeter of Mercury", "pascal", "133.322368"},
	{"centimeter of Mercury", "megapascal", "0.0001333224"},
	{"centimeter of Mercury", "microbar", "1360.432331"},
	{"centimeter of Mercury", "centimeter of Water Column", "13.595098"},
	{"centimeter of Mercury", "decipascal", "1333.223684"},
	{"decipascal", "inch of Mercury", "0.00002952999"},
	{"decipascal", "decapascal", "0.01"},
	{"decipascal", "millibar", "0.001020408"},
	{"decipascal", "pound-force per square inch", "0.00001450377"},
	{"decipascal", "picopascal", "100000000000"},
	{"decipascal", "gigapascal", "0.0000000001"},
	{"decipascal", "centipascal", "10"},
	{"decipascal", "barye", "1"},
	{"decipascal", "millimeter of Water Column", "0.1019716"},
	{"decipascal", "meter of Mercury", "0.00000750062"},
	{"decipascal", "petapascal", "0.0000000000000001"},
	{"decipascal", "bar", "0.000001020408"},
	{"decipascal", "terapascal", "0.0000000000001"},
	{"decipascal", "inch of Water Column", "0.000401865"},
	{"decipascal", "newton per square meter", "0.1"},
	{"decipascal", "millipascal", "100"},
	{"decipascal", "exapascal", "0.0000000000000000001"},
	{"decipascal", "attopascal", "100000000000000000"},
	{"decipascal", "meter of Water Column", "0.0001019716"},
	{"decipascal", "kilopascal", "0.0001"},
	{"decipascal", "torr", "0.000750062"},
	{"decipascal", "micropascal", "100000"},
	{"decipascal", "femtopascal", "100000000000000"},
	{"decipascal", "technical atmosphere", "0.000001019716"},
	{"decipascal", "hectopascal", "0.001"},
	{"decipascal", "nanopascal", "100000000"},
	{"decipascal", "millimeter of Mercury", "0.00750062"},
	{"decipascal", "standard atmosphere", "0.000000986921"},
	{"decipascal", "centibar", "0.0001020408"},
	{"decipascal", "pascal", "0.1"},
	{"decipascal", "megapascal", "0.0000001"},
	{"decipascal", "microbar", "1.020408"},
	{"decipascal", "centimeter of Water Column", "0.01019716"},
	{"decipascal", "centimeter of Mercury", "0.000750062"},
	{"exasecond", "hectosecond", "10000000000000000"},
	{"exasecond", "score", "1584404390.701447"},
	{"exasecond", "microsecond", "999999999999999983222784"},
	{"exasecond", "nanosecond", "999999999999999875848601600"},
	{"exasecond", "femtosecond", "999999999999999945575230987042816"},
	{"exasecond", "decasecond", "100000000000000000"},
	{"exasecond", "gigasecond", "1000000000"},
	{"exasecond", "millisecond", "1000000000000000000000"},
	{"exasecond", "decade", "3168808781.402895"},
	{"exasecond", "second", "1000000000000000000"},
	{"exasecond", "year", "31688087814.02895"},
	{"exasecond", "day", "11574074074074.074219"},
	{"exasecond", "decisecond", "10000000000000000000"},
	{"exasecond", "century", "316880878.140289"},
	{"exasecond", "megasecond", "1000000000000"},
	{"exasecond", "kilosecond", "1000000000000000"},
	{"exasecond", "picosecond", "1000000000000000019884624838656"},
	{"exasecond", "month", "385802469135.80249"},
	{"exasecond", "minute", "16666666666666666"},
	{"exasecond", "millennium", "31688087.814029"},
	{"exasecond", "petasecond", "1000"},
	{"exasecond", "fortnight", "826719576719.576782"},
	{"exasecond", "terasecond", "1000000"},
	{"exasecond", "centisecond", "100000000000000000000"},
	{"exasecond", "hour", "277777777777777.78125"},
	{"exasecond", "planck time", "18552875695732840865584511903108505147301701339185892948967424"},
	{"exasecond", "attosecond", "999999999999999894846684784341549056"},
	{"hectosecond", "exasecond", "0.0000000000000001"},
	{"hectosecond", "score", "0.0000001584404"},
	{"hectosecond", "microsecond", "100000000"},
	{"hectosecond", "nanosecond", "100000000000"},
	{"hectosecond", "femtosecond", "100000000000000000"},
	{"hectosecond", "decasecond", "10"},
	{"hectosecond", "gigasecond", "0.0000001"},
	{"hectosecond", "millisecond", "100000"},
	{"hectosecond", "decade", "0.000000316881"},
	{"hectosecond", "second", "100"},
	{"hectosecond", "year", "0.00000316881"},
	{"hectosecond", "day", "0.001157407"},
	{"hectosecond", "decisecond", "1000"},
	{"hectosecond", "century", "0.0000000316881"},
	{"hectosecond", "megasecond", "0.0001"},
	{"hectosecond", "kilosecond", "0.1"},
	{"hectosecond", "picosecond", "100000000000000"},
	{"hectosecond", "month", "0.0000385802"},
	{"hectosecond", "minute", "1.666667"},
	{"hectosecond", "millennium", "0.00000000316881"},
	{"hectosecond", "petasecond", "0.0000000000001"},
	{"hectosecond", "fortnight", "0.000082672"},
	{"hectosecond", "terasecond", "0.0000000001"},
	{"hectosecond", "centisecond", "10000"},
	{"hectosecond", "hour", "0.02777778"},
	{"hectosecond", "planck time", "1855287569573284018544017630175927559182614528"},
	{"hectosecond", "attosecond", "100000000000000000000"},
	{"score", "exasecond", "0.000000000631152"},
	{"score", "hectosecond", "6311520"},
	{"score", "microsecond", "631152000000000"},
	{"score", "nanosecond", "631152000000000000"},
	{"score", "femtosecond", "631152000000000004194304"},
	{"score", "decasecond", "63115200"},
	{"score", "gigasecond", "0.631152"},
	{"score", "millisecond", "631152000000"},
	{"score", "decade", "2"},
	{"score", "second", "631152000"},
	{"score", "year", "20"},
	{"score", "day", "7305"},
	{"score", "decisecond", "6311520000"},
	{"score", "century", "0.2"},
	{"score", "megasecond", "631.152"},
	{"score", "kilosecond", "631152"},
	{"score", "picosecond", "631152000000000000000"},
	{"score", "month", "243.5"},
	{"score", "minute", "10519200"},
	{"score", "millennium", "0.02"},
	{"score", "petasecond", "0.000000631152"},
	{"score", "fortnight", "521.785714"},
	{"score", "terasecond", "0.000631152"},
	{"score", "centisecond", "63115200000"},
	{"score", "hour", "175320"},
	{"score", "planck time", "11709684601113174059705824502806329134047330520006656"},
	{"score", "attosecond", "631151999999999913999990784"},
	{"microsecond", "exasecond", "0.000000000000000000000001"},
	{"microsecond", "hectosecond", "0.00000001"},
	{"microsecond", "score", "0.000000000000001584404"},
	{"microsecond", "nanosecond", "1000"},
	{"microsecond", "femtosecond", "1000000000"},
	{"microsecond", "decasecond", "0.0000001"},
	{"microsecond", "gigasecond", "0.000000000000001"},
	{"microsecond", "millisecond", "0.001"},
	{"microsecond", "decade", "0.00000000000000316881"},
	{"microsecond", "second", "0.000001"},
	{"microsecond", "year", "0.0000000000000316881"},
	{"microsecond", "day", "0.00000000001157407"},
	{"microsecond", "decisecond", "0.00001"},
	{"microsecond", "century", "0.000000000000000316881"},
	{"microsecond", "megasecond", "0.000000000001"},
	{"microsecond", "kilosecond", "0.000000001"},
	{"microsecond", "picosecond", "1000000"},
	{"microsecond", "month", "0.000000000000385802"},
	{"microsecond", "minute", "0.00000001666667"},
	{"microsecond", "millennium", "0.0000000000000000316881"},
	{"microsecond", "petasecond", "0.000000000000000000001"},
	{"microsecond", "fortnight", "0.00000000000082672"},
	{"microsecond", "terasecond", "0.000000000000000001"},
	{"microsecond", "centisecond", "0.0001"},
	{"microsecond", "hour", "0.0000000002777778"},
	{"microsecond", "planck time", "18552875695732838382593280026855079936"},
	{"microsecond", "attosecond", "999999999999.999878"},
	{"nanosecond", "exasecond", "0.000000000000000000000000001"},
	{"nanosecond", "hectosecond", "0.00000000001"},
	{"nanosecond", "score", "0.000000000000000001584404"},
	{"nanosecond", "microsecond", "0.001"},
	{"nanosecond", "femtosecond", "1000000"},
	{"nanosecond", "decasecond", "0.0000000001"},
	{"nanosecond", "gigasecond", "0.000000000000000001"},
	{"nanosecond", "millisecond", "0.000001"},
	{"nanosecond", "decade", "0.00000000000000000316881"},
	{"nanosecond", "second", "0.000000001"},
	{"nanosecond", "year", "0.0000000000000000316881"},
	{"nanosecond", "day", "0.00000000000001157407"},
	{"nanosecond", "decisecond", "0.00000001"},
	{"nanosecond", "century", "0.000000000000000000316881"},
	{"nanosecond", "megasecond", "0.000000000000001"},
	{"nanosecond", "kilosecond", "0.000000000001"},
	{"nanosecond", "picosecond", "1000"},
	{"nanosecond", "month", "0.000000000000000385802"},
	{"nanosecond", "minute", "0.00000000001666667"},
	{"nanosecond", "millennium", "0.0000000000000000000316881"},
	{"nanosecond", "petasecond", "0.000000000000000000000001"},
	{"nanosecond", "fortnight", "0.00000000000000082672"},
	{"nanosecond", "terasecond", "0.000000000000000000001"},
	{"nanosecond", "centisecond", "0.0000001"},
	{"nanosecond", "hour", "0.0000000000002777778"},
	{"nanosecond", "planck time", "18552875695732840301054663692648448"},
	{"nanosecond", "attosecond", "1000000000"},
	{"femtosecond", "exasecond", "0.000000000000000000000000000000001"},
	{"femtosecond", "hectosecond", "0.00000000000000001"},
	{"femtosecond", "score", "0.000000000000000000000001584404"},
	{"femtosecond", "microsecond", "0.000000001"},
	{"femtosecond", "nanosecond", "0.000001"},
	{"femtosecond", "decasecond", "0.0000000000000001"},
	{"femtosecond", "gigasecond", "0.000000000000000000000001"},
	{"femtosecond", "millisecond", "0.000000000001"},
	{"femtosecond", "decade", "0.00000000000000000000000316881"},
	{"femtosecond", "second", "0.000000000000001"},
	{"femtosecond", "year", "0.0000000000000000000000316881"},
	{"femtosecond", "day", "0.00000000000000000001157407"},
	{"femtosecond", "decisecond", "0.00000000000001"},
	{"femtosecond", "century", "0.000000000000000000000000316881"},
	{"femtosecond", "megasecond", "0.000000000000000000001"},
	{"femtosecond", "kilosecond", "0.000000000000000001"},
	{"femtosecond", "picosecond", "0.001"},
	{"femtosecond", "month", "0.000000000000000000000385802"},
	{"femtosecond", "minute", "0.00000000000000001666667"},
	{"femtosecond", "millennium", "0.0000000000000000000000000316881"},
	{"femtosecond", "petasecond", "0.000000000000000000000000000001"},
	{"femtosecond", "fortnight", "0.00000000000000000000082672"},
	{"femtosecond", "terasecond", "0.000000000000000000000000001"},
	{"femtosecond", "centisecond", "0.0000000000001"},
	{"femtosecond", "hour", "0.0000000000000000002777778"},
	{"femtosecond", "planck time", "18552875695732842406346752000"},
	{"femtosecond", "attosecond", "1000"},
	{"decasecond", "exasecond", "0.00000000000000001"},
	{"decasecond", "hectosecond", "0.1"},
	{"decasecond", "score", "0.00000001584404"},
	{"decasecond", "microsecond", "10000000"},
	{"decasecond", "nanosecond", "10000000000"},
	{"decasecond", "femtosecond", "10000000000000000"},
	{"decasecond", "gigasecond", "0.00000001"},
	{"decasecond", "millisecond", "10000"},
	{"decasecond", "decade", "0.0000000316881"},
	{"decasecond", "second", "10"},
	{"decasecond", "year", "0.000000316881"},
	{"decasecond", "day", "0.0001157407"},
	{"decasecond", "decisecond", "100"},
	{"decasecond", "century", "0.00000000316881"},
	{"decasecond", "megasecond", "0.00001"},
	{"decasecond", "kilosecond", "0.01"},
	{"decasecond", "picosecond", "10000000000000"},
	{"decasecond", "month", "0.00000385802"},
	{"decasecond", "minute", "0.1666667"},
	{"decasecond", "millennium", "0.000000000316881"},
	{"decasecond", "petasecond", "0.00000000000001"},
	{"decasecond", "fortnight", "0.0000082672"},
	{"decasecond", "terasecond", "0.00000000001"},
	{"decasecond", "centisecond", "1000"},
	{"decasecond", "hour", "0.002777778"},
	{"decasecond", "planck time", "185528756957328417700034265870460274627051520"},
	{"decasecond", "attosecond", "10000000000000000000"},
	{"gigasecond", "exasecond", "0.000000001"},
	{"gigasecond", "hectosecond", "10000000"},
	{"gigasecond", "score", "1.584404"},
	{"gigasecond", "microsecond", "1000000000000000"},
	{"gigasecond", "nanosecond", "1000000000000000000"},
	{"gigasecond", "femtosecond", "999999999999999983222784"},
	{"gigasecond", "decasecond", "100000000"},
	{"gigasecond", "millisecond", "1000000000000"},
	{"gigasecond", "decade", "3.168809"},
	{"gigasecond", "second", "1000000000"},
	{"gigasecond", "year", "31.688088"},
	{"gigasecond", "day", "11574.074074"},
	{"gigasecond", "decisecond", "10000000000"},
	{"gigasecond", "century", "0.316881"},
	{"gigasecond", "megasecond", "1000"},
	{"gigasecond", "kilosecond", "1000000"},
	{"gigasecond", "picosecond", "1000000000000000000000"},
	{"gigasecond", "month", "385.802469"},
	{"gigasecond", "minute", "16666666.666667"},
	{"gigasecond", "millennium", "0.0316881"},
	{"gigasecond", "petasecond", "0.000001"},
	{"gigasecond", "fortnight", "826.719577"},
	{"gigasecond", "terasecond", "0.001"},
	{"gigasecond", "centisecond", "100000000000"},
	{"gigasecond", "hour", "277777.777778"},
	{"gigasecond", "planck time", "18552875695732841211446148512082675657621798834405376"},
	{"gigasecond", "attosecond", "999999999999999875848601600"},
	{"millisecond", "exasecond", "0.000000000000000000001"},
	{"millisecond", "hectosecond", "0.00001"},
	{"millisecond", "score", "0.000000000001584404"},
	{"millisecond", "microsecond", "1000"},
	{"millisecond", "nanosecond", "1000000"},
	{"millisecond", "femtosecond", "1000000000000"},
	{"millisecond", "decasecond", "0.0001"},
	{"millisecond", "gigasecond", "0.000000000001"},
	{"millisecond", "decade", "0.00000000000316881"},
	{"millisecond", "second", "0.001"},
	{"millisecond", "year", "0.0000000000316881"},
	{"millisecond", "day", "0.00000001157407"},
	{"millisecond", "decisecond", "0.01"},
	{"millisecond", "century", "0.000000000000316881"},
	{"millisecond", "megasecond", "0.000000001"},
	{"millisecond", "kilosecond", "0.000001"},
	{"millisecond", "picosecond", "1000000000"},
	{"millisecond", "month", "0.000000000385802"},
	{"millisecond", "minute", "0.00001666667"},
	{"millisecond", "millennium", "0.0000000000000316881"},
	{"millisecond", "petasecond", "0.000000000000000001"},
	{"millisecond", "fortnight", "0.00000000082672"},
	{"millisecond", "terasecond", "0.000000000000001"},
	{"millisecond", "centisecond", "0.1"},
	{"millisecond", "hour", "0.0000002777778"},
	{"millisecond", "planck time", "18552875695732839893750554545141548318720"},
	{"millisecond", "attosecond", "1000000000000000"},
	{"decade", "exasecond", "0.000000000315576"},
	{"decade", "hectosecond", "3155760"},
	{"decade", "score", "0.5"},
	{"decade", "microsecond", "315576000000000"},
	{"decade", "nanosecond", "315576000000000000"},
	{"decade", "femtosecond", "315576000000000002097152"},
	{"decade", "decasecond", "31557600"},
	{"decade", "gigasecond", "0.315576"},
	{"decade", "millisecond", "315576000000"},
	{"decade", "second", "315576000"},
	{"decade", "year", "10"},
	{"decade", "day", "3652.5"},
	{"decade", "decisecond", "3155760000"},
	{"decade", "century", "0.1"},
	{"decade", "megasecond", "315.576"},
	{"decade", "kilosecond", "315576"},
	{"decade", "picosecond", "315576000000000000000"},
	{"decade", "month", "121.75"},
	{"decade", "minute", "5259600"},
	{"decade", "millennium", "0.01"},
	{"decade", "petasecond", "0.000000315576"},
	{"decade", "fortnight", "260.892857"},
	{"decade", "terasecond", "0.000315576"},
	{"decade", "centisecond", "31557600000"},
	{"decade", "hour", "87660"},
	{"decade", "planck time", "5854842300556587029852912251403164567023665260003328"},
	{"decade", "attosecond", "315575999999999956999995392"},
	{"second", "exasecond", "0.000000000000000001"},
	{"second", "hectosecond", "0.01"},
	{"second", "score", "0.000000001584404"},
	{"second", "microsecond", "1000000"},
	{"second", "nanosecond", "1000000000"},
	{"second", "femtosecond", "999999999999999.875"},
	{"second", "decasecond", "0.1"},
	{"second", "gigasecond", "0.000000001"},
	{"second", "millisecond", "1000"},
	{"second", "decade", "0.00000000316881"},
	{"second", "year", "0.0000000316881"},
	{"second", "day", "0.00001157407"},
	{"second", "decisecond", "10"},
	{"second", "century", "0.000000000316881"},
	{"second", "megasecond", "0.000001"},
	{"second", "kilosecond", "0.001"},
	{"second", "picosecond", "1000000000000"},
	{"second", "month", "0.000000385802"},
	{"second", "minute", "0.01666667"},
	{"second", "millennium", "0.0000000000316881"},
	{"second", "petasecond", "0.000000000000001"},
	{"second", "fortnight", "0.00000082672"},
	{"second", "terasecond", "0.000000000001"},
	{"second", "centisecond", "100"},
	{"second", "hour", "0.0002777778"},
	{"second", "planck time", "18552875695732839294123348016285477664456704"},
	{"second", "attosecond", "999999999999999872"},
	{"year", "exasecond", "0.0000000000315576"},
	{"year", "hectosecond", "315576"},
	{"year", "score", "0.05"},
	{"year", "microsecond", "31557600000000"},
	{"year", "nanosecond", "31557600000000000"},
	{"year", "femtosecond", "31557599999999996854272"},
	{"year", "decasecond", "3155760"},
	{"year", "gigasecond", "0.0315576"},
	{"year", "millisecond", "31557600000"},
	{"year", "decade", "0.1"},
	{"year", "second", "31557600"},
	{"year", "day", "365.25"},
	{"year", "decisecond", "315576000"},
	{"year", "century", "0.01"},
	{"year", "megasecond", "31.5576"},
	{"year", "kilosecond", "31557.6"},
	{"year", "picosecond", "31557600000000000000"},
	{"year", "month", "12.175"},
	{"year", "minute", "525960"},
	{"year", "millennium", "0.001"},
	{"year", "petasecond", "0.0000000315576"},
	{"year", "fortnight", "26.089286"},
	{"year", "terasecond", "0.0000315576"},
	{"year", "centisecond", "3155760000"},
	{"year", "hour", "8766"},
	{"year", "planck time", "585484230055658719600641172451764867999954779504640"},
	{"year", "attosecond", "31557599999999999135973376"},
	{"day", "exasecond", "0.0000000000000864"},
	{"day", "hectosecond", "864"},
	{"day", "score", "0.0001368925"},
	{"day", "microsecond", "86400000000"},
	{"day", "nanosecond", "86400000000000"},
	{"day", "femtosecond", "86400000000000000000"},
	{"day", "decasecond", "8640"},
	{"day", "gigasecond", "0.0000864"},
	{"day", "millisecond", "86400000"},
	{"day", "decade", "0.0002737851"},
	{"day", "second", "86400"},
	{"day", "year", "0.002737851"},
	{"day", "decisecond", "864000"},
	{"day", "century", "0.00002737851"},
	{"day", "megasecond", "0.0864"},
	{"day", "kilosecond", "86.4"},
	{"day", "picosecond", "86400000000000000"},
	{"day", "month", "0.0333333"},
	{"day", "minute", "1440"},
	{"day", "millennium", "0.000002737851"},
	{"day", "petasecond", "0.0000000000864"},
	{"day", "fortnight", "0.0714286"},
	{"day", "terasecond", "0.0000000864"},
	{"day", "centisecond", "8640000"},
	{"day", "hour", "24"},
	{"day", "planck time", "1602968460111317341316007223342825351265650737152"},
	{"day", "attosecond", "86400000000000000000000"},
	{"decisecond", "exasecond", "0.0000000000000000001"},
	{"decisecond", "hectosecond", "0.001"},
	{"decisecond", "score", "0.0000000001584404"},
	{"decisecond", "microsecond", "100000"},
	{"decisecond", "nanosecond", "100000000"},
	{"decisecond", "femtosecond", "100000000000000"},
	{"decisecond", "decasecond", "0.01"},
	{"decisecond", "gigasecond", "0.0000000001"},
	{"decisecond", "millisecond", "100"},
	{"decisecond", "decade", "0.000000000316881"},
	{"decisecond", "second", "0.1"},
	{"decisecond", "year", "0.00000000316881"},
	{"decisecond", "day", "0.000001157407"},
	{"decisecond", "century", "0.0000000000316881"},
	{"decisecond", "megasecond", "0.0000001"},
	{"decisecond", "kilosecond", "0.0001"},
	{"decisecond", "picosecond", "100000000000"},
	{"decisecond", "month", "0.0000000385802"},
	{"decisecond", "minute", "0.001666667"},
	{"decisecond", "millennium", "0.00000000000316881"},
	{"decisecond", "petasecond", "0.0000000000000001"},
	{"decisecond", "fortnight", "0.000000082672"},
	{"decisecond", "terasecond", "0.0000000000001"},
	{"decisecond", "centisecond", "10"},
	{"decisecond", "hour", "0.00002777778"},
	{"decisecond", "planck time", "1855287569573284115103340694435589001314304"},
	{"decisecond", "attosecond", "100000000000000000"},
	{"century", "exasecond", "0.00000000315576"},
	{"century", "hectosecond", "31557600"},
	{"century", "score", "5"},
	{"century", "microsecond", "3155760000000000"},
	{"century", "nanosecond", "3155760000000000000"},
	{"century", "femtosecond", "3155760000000000020971520"},
	{"century", "decasecond", "315576000"},
	{"century", "gigasecond", "3.15576"},
	{"century", "millisecond", "3155760000000"},
	{"century", "decade", "10"},
	{"century", "second", "3155760000"},
	{"century", "year", "100"},
	{"century", "day", "36525"},
	{"century", "decisecond", "31557600000"},
	{"century", "megasecond", "3155.76"},
	{"century", "kilosecond", "3155760"},
	{"century", "picosecond", "3155760000000000000000"},
	{"century", "month", "1217.5"},
	{"century", "minute", "52596000"},
	{"century", "millennium", "0.1"},
	{"century", "petasecond", "0.00000315576"},
	{"century", "fortnight", "2608.928571"},
	{"century", "terasecond", "0.00315576"},
	{"century", "centisecond", "315576000000"},
	{"century", "hour", "876600"},
	{"century", "planck time", "58548423005565871627757118298947518574043712880377856"},
	{"century", "attosecond", "3155759999999999982316814336"},
	{"megasecond", "exasecond", "0.000000000001"},
	{"megasecond", "hectosecond", "10000"},
	{"megasecond", "score", "0.001584404"},
	{"megasecond", "microsecond", "1000000000000"},
	{"megasecond", "nanosecond", "1000000000000000"},
	{"megasecond", "femtosecond", "999999999999999868928"},
	{"megasecond", "decasecond", "100000"},
	{"megasecond", "gigasecond", "0.001"},
	{"megasecond", "millisecond", "1000000000"},
	{"megasecond", "decade", "0.00316881"},
	{"megasecond", "second", "1000000"},
	{"megasecond", "year", "0.0316881"},
	{"megasecond", "day", "11.574074"},
	{"megasecond", "decisecond", "10000000"},
	{"megasecond", "century", "0.000316881"},
	{"megasecond", "kilosecond", "1000"},
	{"megasecond", "picosecond", "1000000000000000000"},
	{"megasecond", "month", "0.385802"},
	{"megasecond", "minute", "16666.666667"},
	{"megasecond", "millennium", "0.0000316881"},
	{"megasecond", "petasecond", "0.000000001"},
	{"megasecond", "fortnight", "0.82672"},
	{"megasecond", "terasecond", "0.000001"},
	{"megasecond", "centisecond", "100000000"},
	{"megasecond", "hour", "277.777778"},
	{"megasecond", "planck time", "18552875695732840920677524434132328459914004398080"},
	{"megasecond", "attosecond", "999999999999999983222784"},
	{"kilosecond", "exasecond", "0.000000000000001"},
	{"kilosecond", "hectosecond", "10"},
	{"kilosecond", "score", "0.000001584404"},
	{"kilosecond", "microsecond", "1000000000"},
	{"kilosecond", "nanosecond", "999999999999.999878"},
	{"kilosecond", "femtosecond", "999999999999999872"},
	{"kilosecond", "decasecond", "100"},
	{"kilosecond", "gigasecond", "0.000001"},
	{"kilosecond", "millisecond", "1000000"},
	{"kilosecond", "decade", "0.00000316881"},
	{"kilosecond", "second", "1000"},
	{"kilosecond", "year", "0.0000316881"},
	{"kilosecond", "day", "0.01157407"},
	{"kilosecond", "decisecond", "10000"},
	{"kilosecond", "century", "0.000000316881"},
	{"kilosecond", "megasecond", "0.001"},
	{"kilosecond", "picosecond", "1000000000000000"},
	{"kilosecond", "month", "0.000385802"},
	{"kilosecond", "minute", "16.666667"},
	{"kilosecond", "millennium", "0.0000000316881"},
	{"kilosecond", "petasecond", "0.000000000001"},
	{"kilosecond", "fortnight", "0.00082672"},
	{"kilosecond", "terasecond", "0.000000001"},
	{"kilosecond", "centisecond", "100000"},
	{"kilosecond", "hour", "0.2777778"},
	{"kilosecond", "planck time", "18552875695732840819265476415873976340177747968"},
	{"kilosecond", "attosecond", "999999999999999868928"},
	{"picosecond", "exasecond", "0.000000000000000000000000000001"},
	{"picosecond", "hectosecond", "0.00000000000001"},
	{"picosecond", "score", "0.000000000000000000001584404"},
	{"picosecond", "microsecond", "0.000001"},
	{"picosecond", "nanosecond", "0.001"},
	{"picosecond", "femtosecond", "1000"},
	{"picosecond", "decasecond", "0.0000000000001"},
	{"picosecond", "gigasecond", "0.000000000000000000001"},
	{"picosecond", "millisecond", "0.000000001"},
	{"picosecond", "decade", "0.00000000000000000000316881"},
	{"picosecond", "second", "0.000000000001"},
	{"picosecond", "year", "0.0000000000000000000316881"},
	{"picosecond", "day", "0.00000000000000001157407"},
	{"picosecond", "decisecond", "0.00000000001"},
	{"picosecond", "century", "0.000000000000000000000316881"},
	{"picosecond", "megasecond", "0.000000000000000001"},
	{"picosecond", "kilosecond", "0.000000000000001"},
	{"picosecond", "month", "0.000000000000000000385802"},
	{"picosecond", "minute", "0.00000000000001666667"},
	{"picosecond", "millennium", "0.0000000000000000000000316881"},
	{"picosecond", "petasecond", "0.000000000000000000000000001"},
	{"picosecond", "fortnight", "0.00000000000000000082672"},
	{"picosecond", "terasecond", "0.000000000000000000000001"},
	{"picosecond", "centisecond", "0.0000000001"},
	{"picosecond", "hour", "0.0000000000000002777778"},
	{"picosecond", "planck time", "18552875695732840805457821958144"},
	{"picosecond", "attosecond", "1000000"},
	{"month", "exasecond", "0.000000000002592"},
	{"month", "hectosecond", "25920"},
	{"month", "score", "0.00410678"},
	{"month", "microsecond", "2592000000000"},
	{"month", "nanosecond", "2592000000000000"},
	{"month", "femtosecond", "2592000000000000000000"},
	{"month", "decasecond", "259200"},
	{"month", "gigasecond", "0.002592"},
	{"month", "millisecond", "2592000000"},
	{"month", "decade", "0.00821355"},
	{"month", "second", "2592000"},
	{"month", "year", "0.0821355"},
	{"month", "day", "30"},
	{"month", "decisecond", "25920000"},
	{"month", "century", "0.000821355"},
	{"month", "megasecond", "2.592"},
	{"month", "kilosecond", "2592"},
	{"month", "picosecond", "2592000000000000000"},
	{"month", "minute", "43200"},
	{"month", "millennium", "0.0000821355"},
	{"month", "petasecond", "0.000000002592"},
	{"month", "fortnight", "2.142857"},
	{"month", "terasecond", "0.000002592"},
	{"month", "centisecond", "259200000"},
	{"month", "hour", "720"},
	{"month", "planck time", "48089053803339522835628645967698574803217686724608"},
	{"month", "attosecond", "2591999999999999664455680"},
	{"minute", "exasecond", "0.00000000000000006"},
	{"minute", "hectosecond", "0.6"},
	{"minute", "score", "0.0000000950643"},
	{"minute", "microsecond", "60000000"},
	{"minute", "nanosecond", "60000000000"},
	{"minute", "femtosecond", "59999999999999992"},
	{"minute", "decasecond", "6"},
	{"minute", "gigasecond", "0.00000006"},
	{"minute", "millisecond", "60000"},
	{"minute", "decade", "0.0000001901285"},
	{"minute", "second", "60"},
	{"minute", "year", "0.000001901285"},
	{"minute", "day", "0.000694444"},
	{"minute", "decisecond", "600"},
	{"minute", "century", "0.00000001901285"},
	{"minute", "megasecond", "0.00006"},
	{"minute", "kilosecond", "0.06"},
	{"minute", "picosecond", "60000000000000"},
	{"minute", "month", "0.00002314815"},
	{"minute", "millennium", "0.000000001901285"},
	{"minute", "petasecond", "0.00000000000006"},
	{"minute", "fortnight", "0.0000496032"},
	{"minute", "terasecond", "0.00000000006"},
	{"minute", "centisecond", "6000"},
	{"minute", "hour", "0.01666667"},
	{"minute", "planck time", "1113172541743970347743880566694086460674408448"},
	{"minute", "attosecond", "59999999999999991808"},
	{"millennium", "exasecond", "0.0000000315576"},
	{"millennium", "hectosecond", "315576000"},
	{"millennium", "score", "50"},
	{"millennium", "microsecond", "31557600000000000"},
	{"millennium", "nanosecond", "31557600000000000000"},
	{"millennium", "femtosecond", "31557599999999999135973376"},
	{"millennium", "decasecond", "3155760000"},
	{"millennium", "gigasecond", "31.5576"},
	{"millennium", "millisecond", "31557600000000"},
	{"millennium", "decade", "100"},
	{"millennium", "second", "31557600000"},
	{"millennium", "year", "1000"},
	{"millennium", "day", "365250"},
	{"millennium", "decisecond", "315576000000"},
	{"millennium", "century", "10"},
	{"millennium", "megasecond", "31557.6"},
	{"millennium", "kilosecond", "31557600"},
	{"millennium", "picosecond", "31557600000000001048576"},
	{"millennium", "month", "12175"},
	{"millennium", "minute", "525960000"},
	{"millennium", "petasecond", "0.0000315576"},
	{"millennium", "fortnight", "26089.285714"},
	{"millennium", "terasecond", "0.0315576"},
	{"millennium", "centisecond", "3155760000000"},
	{"millennium", "hour", "8766000"},
	{"millennium", "planck time", "585484230055658652474627385313513286357698235347238912"},
	{"millennium", "attosecond", "31557599999999999823168143360"},
	{"petasecond", "exasecond", "0.001"},
	{"petasecond", "hectosecond", "10000000000000"},
	{"petasecond", "score", "1584404.390701"},
	{"petasecond", "microsecond", "1000000000000000000000"},
	{"petasecond", "nanosecond", "999999999999999983222784"},
	{"petasecond", "femtosecond", "999999999999999879147136483328"},
	{"petasecond", "decasecond", "100000000000000"},
	{"petasecond", "gigasecond", "1000000"},
	{"petasecond", "millisecond", "1000000000000000000"},
	{"petasecond", "decade", "3168808.781403"},
	{"petasecond", "second", "1000000000000000"},
	{"petasecond", "year", "31688087.814029"},
	{"petasecond", "day", "11574074074.074074"},
	{"petasecond", "decisecond", "10000000000000000"},
	{"petasecond", "century", "316880.87814"},
	{"petasecond", "megasecond", "1000000000"},
	{"petasecond", "kilosecond", "1000000000000"},
	{"petasecond", "picosecond", "1000000000000000013287555072"},
	{"petasecond", "month", "385802469.135802"},
	{"petasecond", "minute", "16666666666666.666016"},
	{"petasecond", "millennium", "31688.087814"},
	{"petasecond", "fortnight", "826719576.719577"},
	{"petasecond", "terasecond", "1000"},
	{"petasecond", "centisecond", "100000000000000000"},
	{"petasecond", "hour", "277777777777.777771"},
	{"petasecond", "planck time", "18552875695732841110892709086945359704194602338317869514752"},
	{"petasecond", "attosecond", "999999999999999945575230987042816"},
	{"fortnight", "exasecond", "0.0000000000012096"},
	{"fortnight", "hectosecond", "12096"},
	{"fortnight", "score", "0.001916496"},
	{"fortnight", "microsecond", "1209600000000"},
	{"fortnight", "nanosecond", "1209600000000000"},
	{"fortnight", "femtosecond", "1209600000000000000000"},
	{"fortnight", "decasecond", "120960"},
	{"fortnight", "gigasecond", "0.0012096"},
	{"fortnight", "millisecond", "1209600000"},
	{"fortnight", "decade", "0.00383299"},
	{"fortnight", "second", "1209600"},
	{"fortnight", "year", "0.0383299"},
	{"fortnight", "day", "14"},
	{"fortnight", "decisecond", "12096000"},
	{"fortnight", "century", "0.000383299"},
	{"fortnight", "megasecond", "1.2096"},
	{"fortnight", "kilosecond", "1209.6"},
	{"fortnight", "picosecond", "1209600000000000000"},
	{"fortnight", "month", "0.466667"},
	{"fortnight", "minute", "20160"},
	{"fortnight", "millennium", "0.0000383299"},
	{"fortnight", "petasecond", "0.0000000012096"},
	{"fortnight", "terasecond", "0.0000012096"},
	{"fortnight", "centisecond", "120960000"},
	{"fortnight", "hour", "336"},
	{"fortnight", "planck time", "22441558441558442778424101126799554917719110320128"},
	{"fortnight", "attosecond", "1209599999999999932891136"},
	{"terasecond", "exasecond", "0.000001"},
	{"terasecond", "hectosecond", "10000000000"},
	{"terasecond", "score", "1584.404391"},
	{"terasecond", "microsecond", "1000000000000000000"},
	{"terasecond", "nanosecond", "1000000000000000000000"},
	{"terasecond", "femtosecond", "999999999999999875848601600"},
	{"terasecond", "decasecond", "100000000000"},
	{"terasecond", "gigasecond", "1000"},
	{"terasecond", "millisecond", "1000000000000000"},
	{"terasecond", "decade", "3168.808781"},
	{"terasecond", "second", "1000000000000"},
	{"terasecond", "year", "31688.087814"},
	{"terasecond", "day", "11574074.074074"},
	{"terasecond", "decisecond", "10000000000000"},
	{"terasecond", "century", "316.880878"},
	{"terasecond", "megasecond", "1000000"},
	{"terasecond", "kilosecond", "1000000000"},
	{"terasecond", "picosecond", "999999999999999983222784"},
	{"terasecond", "month", "385802.469136"},
	{"terasecond", "minute", "16666666666.666666"},
	{"terasecond", "millennium", "31.688088"},
	{"terasecond", "petasecond", "0.001"},
	{"terasecond", "fortnight", "826719.57672"},
	{"terasecond", "centisecond", "100000000000000"},
	{"terasecond", "hour", "277777777.777778"},
	{"terasecond", "planck time", "18552875695732839956654920491122091636427933929760096256"},
	{"terasecond", "attosecond", "999999999999999879147136483328"},
	{"centisecond", "exasecond", "0.00000000000000000001"},
	{"centisecond", "hectosecond", "0.0001"},
	{"centisecond", "score", "0.00000000001584404"},
	{"centisecond", "microsecond", "10000"},
	{"centisecond", "nanosecond", "10000000"},
	{"centisecond", "femtosecond", "10000000000000"},
	{"centisecond", "decasecond", "0.001"},
	{"centisecond", "gigasecond", "0.00000000001"},
	{"centisecond", "millisecond", "10"},
	{"centisecond", "decade", "0.0000000000316881"},
	{"centisecond", "second", "0.01"},
	{"centisecond", "year", "0.000000000316881"},
	{"centisecond", "day", "0.0000001157407"},
	{"centisecond", "decisecond", "0.1"},
	{"centisecond", "century", "0.00000000000316881"},
	{"centisecond", "megasecond", "0.00000001"},
	{"centisecond", "kilosecond", "0.00001"},
	{"centisecond", "picosecond", "10000000000"},
	{"centisecond", "month", "0.00000000385802"},
	{"centisecond", "minute", "0.0001666667"},
	{"centisecond", "millennium", "0.000000000000316881"},
	{"centisecond", "petasecond", "0.00000000000000001"},
	{"centisecond", "fortnight", "0.0000000082672"},
	{"centisecond", "terasecond", "0.00000000000001"},
	{"centisecond", "hour", "0.000002777778"},
	{"centisecond", "planck time", "185528756957328403773208823909932182011904"},
	{"centisecond", "attosecond", "10000000000000000"},
	{"hour", "exasecond", "0.0000000000000036"},
	{"hour", "hectosecond", "36"},
	{"hour", "score", "0.00000570386"},
	{"hour", "microsecond", "3600000000"},
	{"hour", "nanosecond", "3600000000000"},
	{"hour", "femtosecond", "3599999999999999488"},
	{"hour", "decasecond", "360"},
	{"hour", "gigasecond", "0.0000036"},
	{"hour", "millisecond", "3600000"},
	{"hour", "decade", "0.00001140771"},
	{"hour", "second", "3600"},
	{"hour", "year", "0.0001140771"},
	{"hour", "day", "0.0416667"},
	{"hour", "decisecond", "36000"},
	{"hour", "century", "0.000001140771"},
	{"hour", "megasecond", "0.0036"},
	{"hour", "kilosecond", "3.6"},
	{"hour", "picosecond", "3600000000000000"},
	{"hour", "month", "0.001388889"},
	{"hour", "minute", "60"},
	{"hour", "millennium", "0.0000001140771"},
	{"hour", "petasecond", "0.0000000000036"},
	{"hour", "fortnight", "0.00297619"},
	{"hour", "terasecond", "0.0000000036"},
	{"hour", "centisecond", "360000"},
	{"hour", "planck time", "66790352504638225935235234914562793627277328384"},
	{"hour", "attosecond", "3600000000000000000000"},
	{"planck time", "exasecond", "0.0000000000000000000000000000000000000000000000000000000000000539"},
	{"planck time", "hectosecond", "0.000000000000000000000000000000000000000000000539"},
	{"planck time", "score", "0.0000000000000000000000000000000000000000000000000000853994"},
	{"planck time", "microsecond", "0.0000000000000000000000000000000000000539"},
	{"planck time", "nanosecond", "0.0000000000000000000000000000000000539"},
	{"planck time", "femtosecond", "0.0000000000000000000000000000539"},
	{"planck time", "decasecond", "0.00000000000000000000000000000000000000000000539"},
	{"planck time", "gigasecond", "0.0000000000000000000000000000000000000000000000000000539"},
	{"planck time", "millisecond", "0.0000000000000000000000000000000000000000539"},
	{"planck time", "decade", "0.0000000000000000000000000000000000000000000000000001707988"},
	{"planck time", "second", "0.0000000000000000000000000000000000000000000539"},
	{"planck time", "year", "0.000000000000000000000000000000000000000000000000001707988"},
	{"planck time", "day", "0.000000000000000000000000000000000000000000000000623843"},
	{"planck time", "decisecond", "0.000000000000000000000000000000000000000000539"},
	{"planck time", "century", "0.00000000000000000000000000000000000000000000000000001707988"},
	{"planck time", "megasecond", "0.0000000000000000000000000000000000000000000000000539"},
	{"planck time", "kilosecond", "0.0000000000000000000000000000000000000000000000539"},
	{"planck time", "picosecond", "0.0000000000000000000000000000000539"},
	{"planck time", "month", "0.00000000000000000000000000000000000000000000000002079475"},
	{"planck time", "minute", "0.000000000000000000000000000000000000000000000898333"},
	{"planck time", "millennium", "0.000000000000000000000000000000000000000000000000000001707988"},
	{"planck time", "petasecond", "0.0000000000000000000000000000000000000000000000000000000000539"},
	{"planck time", "fortnight", "0.0000000000000000000000000000000000000000000000000445602"},
	{"planck time", "terasecond", "0.0000000000000000000000000000000000000000000000000000000539"},
	{"planck time", "centisecond", "0.00000000000000000000000000000000000000000539"},
	{"planck time", "hour", "0.00000000000000000000000000000000000000000000001497222"},
	{"planck time", "attosecond", "0.0000000000000000000000000539"},
	{"attosecond", "exasecond", "0.000000000000000000000000000000000001"},
	{"attosecond", "hectosecond", "0.00000000000000000001"},
	{"attosecond", "score", "0.000000000000000000000000001584404"},
	{"attosecond", "microsecond", "0.000000000001"},
	{"attosecond", "nanosecond", "0.000000001"},
	{"attosecond", "femtosecond", "0.001"},
	{"attosecond", "decasecond", "0.0000000000000000001"},
	{"attosecond", "gigasecond", "0.000000000000000000000000001"},
	{"attosecond", "millisecond", "0.000000000000001"},
	{"attosecond", "decade", "0.00000000000000000000000000316881"},
	{"attosecond", "second", "0.000000000000000001"},
	{"attosecond", "year", "0.0000000000000000000000000316881"},
	{"attosecond", "day", "0.00000000000000000000001157407"},
	{"attosecond", "decisecond", "0.00000000000000001"},
	{"attosecond", "century", "0.000000000000000000000000000316881"},
	{"attosecond", "megasecond", "0.000000000000000000000001"},
	{"attosecond", "kilosecond", "0.000000000000000000001"},
	{"attosecond", "picosecond", "0.000001"},
	{"attosecond", "month", "0.000000000000000000000000385802"},
	{"attosecond", "minute", "0.00000000000000000001666667"},
	{"attosecond", "millennium", "0.0000000000000000000000000000316881"},
	{"attosecond", "petasecond", "0.000000000000000000000000000000001"},
	{"attosecond", "fortnight", "0.00000000000000000000000082672"},
	{"attosecond", "terasecond", "0.000000000000000000000000000001"},
	{"attosecond", "centisecond", "0.0000000000000001"},
	{"attosecond", "hour", "0.0000000000000000000002777778"},
	{"attosecond", "planck time", "18552875695732842406346752"},
	{"femtoliter", "megaliter", "0.000000000000000000001"},
	{"femtoliter", "petaliter", "0.000000000000000000000000000001"},
	{"femtoliter", "teraliter", "0.000000000000000000000000001"},
	{"femtoliter", "decaliter", "0.0000000000000001"},
	{"femtoliter", "kiloliter", "0.000000000000000001"},
	{"femtoliter", "attoliter", "1000"},
	{"femtoliter", "centiliter", "0.0000000000001"},
	{"femtoliter", "liter", "0.000000000000001"},
	{"femtoliter", "exaliter", "0.000000000000000000000000000000001"},
	{"femtoliter", "fluid gallon", "0.0000000000000002641721"},
	{"femtoliter", "picoliter", "0.001"},
	{"femtoliter", "milliliter", "0.000000000001"},
	{"femtoliter", "gallon", "0.0000000000000002199692"},
	{"femtoliter", "hectoliter", "0.00000000000000001"},
	{"femtoliter", "fluid quart", "0.000000000000001056688"},
	{"femtoliter", "nanoliter", "0.000001"},
	{"femtoliter", "quart", "0.000000000000000879877"},
	{"femtoliter", "fluid pint", "0.000000000000002113376"},
	{"femtoliter", "customary fluid ounce", "0.000000000000033814"},
	{"femtoliter", "pint", "0.000000000000001759754"},
	{"femtoliter", "microliter", "0.000000001"},
	{"femtoliter", "deciliter", "0.00000000000001"},
	{"femtoliter", "gigaliter", "0.000000000000000000000001"},
	{"femtoliter", "fluid ounce", "0.0000000000000351951"},
	{"megaliter", "femtoliter", "999999999999999868928"},
	{"megaliter", "petaliter", "0.000000001"},
	{"megaliter", "teraliter", "0.000001"},
	{"megaliter", "decaliter", "100000"},
	{"megaliter", "kiloliter", "1000"},
	{"megaliter", "attoliter", "999999999999999983222784"},
	{"megaliter", "centiliter", "100000000"},
	{"megaliter", "liter", "1000000"},
	{"megaliter", "exaliter", "0.000000000001"},
	{"megaliter", "fluid gallon", "264172.052358"},
	{"megaliter", "picoliter", "1000000000000000000"},
	{"megaliter", "milliliter", "1000000000"},
	{"megaliter", "gallon", "219969.248299"},
	{"megaliter", "hectoliter", "10000"},
	{"megaliter", "fluid quart", "1056688.209433"},
	{"megaliter", "nanoliter", "1000000000000000"},
	{"megaliter", "quart", "879876.993196"},
	{"megaliter", "fluid pint", "2113376.418865"},
	{"megaliter", "customary fluid ounce", "33814022.701843"},
	{"megaliter", "pint", "1759753.986393"},
	{"megaliter", "microliter", "1000000000000"},
	{"megaliter", "deciliter", "10000000"},
	{"megaliter", "gigaliter", "0.001"},
	{"megaliter", "fluid ounce", "35195079.727854"},
	{"petaliter", "femtoliter", "999999999999999879147136483328"},
	{"petaliter", "megaliter", "1000000000"},
	{"petaliter", "teraliter", "1000"},
	{"petaliter", "decaliter", "100000000000000"},
	{"petaliter", "kiloliter", "1000000000000"},
	{"petaliter", "attoliter", "999999999999999945575230987042816"},
	{"petaliter", "centiliter", "100000000000000000"},
	{"petaliter", "liter", "1000000000000000"},
	{"petaliter", "exaliter", "0.001"},
	{"petaliter", "fluid gallon", "264172052358148.4375"},
	{"petaliter", "picoliter", "1000000000000000013287555072"},
	{"petaliter", "milliliter", "1000000000000000000"},
	{"petaliter", "gallon", "219969248299087.78125"},
	{"petaliter", "hectoliter", "10000000000000"},
	{"petaliter", "fluid quart", "1056688209432593.75"},
	{"petaliter", "nanoliter", "999999999999999983222784"},
	{"petaliter", "quart", "879876993196351.125"},
	{"petaliter", "fluid pint", "2113376418865187.5"},
	{"petaliter", "customary fluid ounce", "33814022701843000"},
	{"petaliter", "pint", "1759753986392702.25"},
	{"petaliter", "microliter", "1000000000000000000000"},
	{"petaliter", "deciliter", "10000000000000000"},
	{"petaliter", "gigaliter", "1000000"},
	{"petaliter", "fluid ounce", "35195079727854048"},
	{"teraliter", "femtoliter", "999999999999999875848601600"},
	{"teraliter", "megaliter", "1000000"},
	{"teraliter", "petaliter", "0.001"},
	{"teraliter", "decaliter", "100000000000"},
	{"teraliter", "kiloliter", "1000000000"},
	{"teraliter", "attoliter", "999999999999999879147136483328"},
	{"teraliter", "centiliter", "100000000000000"},
	{"teraliter", "liter", "1000000000000"},
	{"teraliter", "exaliter", "0.000001"},
	{"teraliter", "fluid gallon", "264172052358.148438"},
	{"teraliter", "picoliter", "999999999999999983222784"},
	{"teraliter", "milliliter", "1000000000000000"},
	{"teraliter", "gallon", "219969248299.087769"},
	{"teraliter", "hectoliter", "10000000000"},
	{"teraliter", "fluid quart", "1056688209432.59375"},
	{"teraliter", "nanoliter", "1000000000000000000000"},
	{"teraliter", "quart", "879876993196.351074"},
	{"teraliter", "fluid pint", "2113376418865.1875"},
	{"teraliter", "customary fluid ounce", "33814022701843"},
	{"teraliter", "pint", "1759753986392.702148"},
	{"teraliter", "microliter", "1000000000000000000"},
	{"teraliter", "deciliter", "10000000000000"},
	{"teraliter", "gigaliter", "1000"},
	{"teraliter", "fluid ounce", "35195079727854.046875"},
	{"decaliter", "femtoliter", "10000000000000000"},
	{"decaliter", "megaliter", "0.00001"},
	{"decaliter", "petaliter", "0.00000000000001"},
	{"decaliter", "teraliter", "0.00000000001"},
	{"decaliter", "kiloliter", "0.01"},
	{"decaliter", "attoliter", "10000000000000000000"},
	{"decaliter", "centiliter", "1000"},
	{"decaliter", "liter", "10"},
	{"decaliter", "exaliter", "0.00000000000000001"},
	{"decaliter", "fluid gallon", "2.641721"},
	{"decaliter", "picoliter", "10000000000000"},
	{"decaliter", "milliliter", "10000"},
	{"decaliter", "gallon", "2.199692"},
	{"decaliter", "hectoliter", "0.1"},
	{"decaliter", "fluid quart", "10.566882"},
	{"decaliter", "nanoliter", "10000000000"},
	{"decaliter", "quart", "8.79877"},
	{"decaliter", "fluid pint", "21.133764"},
	{"decaliter", "customary fluid ounce", "338.140227"},
	{"decaliter", "pint", "17.59754"},
	{"decaliter", "microliter", "10000000"},
	{"decaliter", "deciliter", "100"},
	{"decaliter", "gigaliter", "0.00000001"},
	{"decaliter", "fluid ounce", "351.950797"},
	{"kiloliter", "femtoliter", "999999999999999872"},
	{"kiloliter", "megaliter", "0.001"},
	{"kiloliter", "petaliter", "0.000000000001"},
	{"kiloliter", "teraliter", "0.000000001"},
	{"kiloliter", "decaliter", "100"},
	{"kiloliter", "attoliter", "999999999999999868928"},
	{"kiloliter", "centiliter", "100000"},
	{"kiloliter", "liter", "1000"},
	{"kiloliter", "exaliter", "0.000000000000001"},
	{"kiloliter", "fluid gallon", "264.172052"},
	{"kiloliter", "picoliter", "1000000000000000"},
	{"kiloliter", "milliliter", "1000000"},
	{"kiloliter", "gallon", "219.969248"},
	{"kiloliter", "hectoliter", "10"},
	{"kiloliter", "fluid quart", "1056.688209"},
	{"kiloliter", "nanoliter", "999999999999.999878"},
	{"kiloliter", "quart", "879.876993"},
	{"kiloliter", "fluid pint", "2113.376419"},
	{"kiloliter", "customary fluid ounce", "33814.022702"},
	{"kiloliter", "pint", "1759.753986"},
	{"kiloliter", "microliter", "1000000000"},
	{"kiloliter", "deciliter", "10000"},
	{"kiloliter", "gigaliter", "0.000001"},
	{"kiloliter", "fluid ounce", "35195.079728"},
	{"attoliter", "femtoliter", "0.001"},
	{"attoliter", "megaliter", "0.000000000000000000000001"},
	{"attoliter", "petaliter", "0.000000000000000000000000000000001"},
	{"attoliter", "teraliter", "0.000000000000000000000000000001"},
	{"attoliter", "decaliter", "0.0000000000000000001"},
	{"attoliter", "kiloliter", "0.000000000000000000001"},
	{"attoliter", "centiliter", "0.0000000000000001"},
	{"attoliter", "liter", "0.000000000000000001"},
	{"attoliter", "exaliter", "0.000000000000000000000000000000000001"},
	{"attoliter", "fluid gallon", "0.0000000000000000002641721"},
	{"attoliter", "picoliter", "0.000001"},
	{"attoliter", "milliliter", "0.000000000000001"},
	{"attoliter", "gallon", "0.0000000000000000002199692"},
	{"attoliter", "hectoliter", "0.00000000000000000001"},
	{"attoliter", "fluid quart", "0.000000000000000001056688"},
	{"attoliter", "nanoliter", "0.000000001"},
	{"attoliter", "quart", "0.000000000000000000879877"},
	{"attoliter", "fluid pint", "0.000000000000000002113376"},
	{"attoliter", "customary fluid ounce", "0.000000000000000033814"},
	{"attoliter", "pint", "0.000000000000000001759754"},
	{"attoliter", "microliter", "0.000000000001"},
	{"attoliter", "deciliter", "0.00000000000000001"},
	{"attoliter", "gigaliter", "0.000000000000000000000000001"},
	{"attoliter", "fluid ounce", "0.0000000000000000351951"},
	{"centiliter", "femtoliter", "10000000000000"},
	{"centiliter", "megaliter", "0.00000001"},
	{"centiliter", "petaliter", "0.00000000000000001"},
	{"centiliter", "teraliter", "0.00000000000001"},
	{"centiliter", "decaliter", "0.001"},
	{"centiliter", "kiloliter", "0.00001"},
	{"centiliter", "attoliter", "10000000000000000"},
	{"centiliter", "liter", "0.01"},
	{"centiliter", "exaliter", "0.00000000000000000001"},
	{"centiliter", "fluid gallon", "0.002641721"},
	{"centiliter", "picoliter", "10000000000"},
	{"centiliter", "milliliter", "10"},
	{"centiliter", "gallon", "0.002199692"},
	{"centiliter", "hectoliter", "0.0001"},
	{"centiliter", "fluid quart", "0.01056688"},
	{"centiliter", "nanoliter", "10000000"},
	{"centiliter", "quart", "0.00879877"},
	{"centiliter", "fluid pint", "0.02113376"},
	{"centiliter", "customary fluid ounce", "0.33814"},
	{"centiliter", "pint", "0.01759754"},
	{"centiliter", "microliter", "10000"},
	{"centiliter", "deciliter", "0.1"},
	{"centiliter", "gigaliter", "0.00000000001"},
	{"centiliter", "fluid ounce", "0.351951"},
	{"liter", "femtoliter", "999999999999999.875"},
	{"liter", "megaliter", "0.000001"},
	{"liter", "petaliter", "0.000000000000001"},
	{"liter", "teraliter", "0.000000000001"},
	{"liter", "decaliter", "0.1"},
	{"liter", "kiloliter", "0.001"},
	{"liter", "attoliter", "999999999999999872"},
	{"liter", "centiliter", "100"},
	{"liter", "exaliter", "0.000000000000000001"},
	{"liter", "fluid gallon", "0.2641721"},
	{"liter", "picoliter", "1000000000000"},
	{"liter", "milliliter", "1000"},
	{"liter", "gallon", "0.2199692"},
	{"liter", "hectoliter", "0.01"},
	{"liter", "fluid quart", "1.056688"},
	{"liter", "nanoliter", "1000000000"},
	{"liter", "quart", "0.879877"},
	{"liter", "fluid pint", "2.113376"},
	{"liter", "customary fluid ounce", "33.814023"},
	{"liter", "pint", "1.759754"},
	{"liter", "microliter", "1000000"},
	{"liter", "deciliter", "10"},
	{"liter", "gigaliter", "0.000000001"},
	{"liter", "fluid ounce", "35.19508"},
	{"exaliter", "femtoliter", "999999999999999945575230987042816"},
	{"exaliter", "megaliter", "1000000000000"},
	{"exaliter", "petaliter", "1000"},
	{"exaliter", "teraliter", "1000000"},
	{"exaliter", "decaliter", "100000000000000000"},
	{"exaliter", "kiloliter", "1000000000000000"},
	{"exaliter", "attoliter", "999999999999999894846684784341549056"},
	{"exaliter", "centiliter", "100000000000000000000"},
	{"exaliter", "liter", "1000000000000000000"},
	{"exaliter", "fluid gallon", "264172052358148416"},
	{"exaliter", "picoliter", "1000000000000000019884624838656"},
	{"exaliter", "milliliter", "1000000000000000000000"},
	{"exaliter", "gallon", "219969248299087776"},
	{"exaliter", "hectoliter", "10000000000000000"},
	{"exaliter", "fluid quart", "1056688209432593664"},
	{"exaliter", "nanoliter", "999999999999999875848601600"},
	{"exaliter", "quart", "879876993196351104"},
	{"exaliter", "fluid pint", "2113376418865187328"},
	{"exaliter", "customary fluid ounce", "33814022701842997248"},
	{"exaliter", "pint", "1759753986392702208"},
	{"exaliter", "microliter", "999999999999999983222784"},
	{"exaliter", "deciliter", "10000000000000000000"},
	{"exaliter", "gigaliter", "1000000000"},
	{"exaliter", "fluid ounce", "35195079727854047232"},
	{"fluid gallon", "femtoliter", "3785411783999999.5"},
	{"fluid gallon", "megaliter", "0.00000378541"},
	{"fluid gallon", "petaliter", "0.00000000000000378541"},
	{"fluid gallon", "teraliter", "0.00000000000378541"},
	{"fluid gallon", "decaliter", "0.378541"},
	{"fluid gallon", "kiloliter", "0.00378541"},
	{"fluid gallon", "attoliter", "3785411783999999488"},
	{"fluid gallon", "centiliter", "378.541178"},
	{"fluid gallon", "liter", "3.785412"},
	{"fluid gallon", "exaliter", "0.00000000000000000378541"},
	{"fluid gallon", "picoliter", "3785411784000"},
	{"fluid gallon", "milliliter", "3785.411784"},
	{"fluid gallon", "gallon", "0.832674"},
	{"fluid gallon", "hectoliter", "0.0378541"},
	{"fluid gallon", "fluid quart", "4"},
	{"fluid gallon", "nanoliter", "3785411784"},
	{"fluid gallon", "quart", "3.330697"},
	{"fluid gallon", "fluid pint", "8"},
	{"fluid gallon", "customary fluid ounce", "128"},
	{"fluid gallon", "pint", "6.661393"},
	{"fluid gallon", "microliter", "3785411.784"},
	{"fluid gallon", "deciliter", "37.854118"},
	{"fluid gallon", "gigaliter", "0.00000000378541"},
	{"fluid gallon", "fluid ounce", "133.22787"},
	{"picoliter", "femtoliter", "1000"},
	{"picoliter", "megaliter", "0.000000000000000001"},
	{"picoliter", "petaliter", "0.000000000000000000000000001"},
	{"picoliter", "teraliter", "0.000000000000000000000001"},
	{"picoliter", "decaliter", "0.0000000000001"},
	{"picoliter", "kiloliter", "0.000000000000001"},
	{"picoliter", "attoliter", "1000000"},
	{"picoliter", "centiliter", "0.0000000001"},
	{"picoliter", "liter", "0.000000000001"},
	{"picoliter", "exaliter", "0.000000000000000000000000000001"},
	{"picoliter", "fluid gallon", "0.0000000000002641721"},
	{"picoliter", "milliliter", "0.000000001"},
	{"picoliter", "gallon", "0.0000000000002199692"},
	{"picoliter", "hectoliter", "0.00000000000001"},
	{"picoliter", "fluid quart", "0.000000000001056688"},
	{"picoliter", "nanoliter", "0.001"},
	{"picoliter", "quart", "0.000000000000879877"},
	{"picoliter", "fluid pint", "0.000000000002113376"},
	{"picoliter", "customary fluid ounce", "0.000000000033814"},
	{"picoliter", "pint", "0.000000000001759754"},
	{"picoliter", "microliter", "0.000001"},
	{"picoliter", "deciliter", "0.00000000001"},
	{"picoliter", "gigaliter", "0.000000000000000000001"},
	{"picoliter", "fluid ounce", "0.0000000000351951"},
	{"milliliter", "femtoliter", "1000000000000"},
	{"milliliter", "megaliter", "0.000000001"},
	{"milliliter", "petaliter", "0.000000000000000001"},
	{"milliliter", "teraliter", "0.000000000000001"},
	{"milliliter", "decaliter", "0.0001"},
	{"milliliter", "kiloliter", "0.000001"},
	{"milliliter", "attoliter", "1000000000000000"},
	{"milliliter", "centiliter", "0.1"},
	{"milliliter", "liter", "0.001"},
	{"milliliter", "exaliter", "0.000000000000000000001"},
	{"milliliter", "fluid gallon", "0.0002641721"},
	{"milliliter", "picoliter", "1000000000"},
	{"milliliter", "gallon", "0.0002199692"},
	{"milliliter", "hectoliter", "0.00001"},
	{"milliliter", "fluid quart", "0.001056688"},
	{"milliliter", "nanoliter", "1000000"},
	{"milliliter", "quart", "0.000879877"},
	{"milliliter", "fluid pint", "0.002113376"},
	{"milliliter", "customary fluid ounce", "0.033814"},
	{"milliliter", "pint", "0.001759754"},
	{"milliliter", "microliter", "1000"},
	{"milliliter", "deciliter", "0.01"},
	{"milliliter", "gigaliter", "0.000000000001"},
	{"milliliter", "fluid ounce", "0.0351951"},
	{"gallon", "femtoliter", "4546090000000000"},
	{"gallon", "megaliter", "0.00000454609"},
	{"gallon", "petaliter", "0.00000000000000454609"},
	{"gallon", "teraliter", "0.00000000000454609"},
	{"gallon", "decaliter", "0.454609"},
	{"gallon", "kiloliter", "0.00454609"},
	{"gallon", "attoliter", "4546090000000000000"},
	{"gallon", "centiliter", "454.609"},
	{"gallon", "liter", "4.54609"},
	{"gallon", "exaliter", "0.00000000000000000454609"},
	{"gallon", "fluid gallon", "1.20095"},
	{"gallon", "picoliter", "4546090000000.000977"},
	{"gallon", "milliliter", "4546.09"},
	{"gallon", "hectoliter", "0.0454609"},
	{"gallon", "fluid quart", "4.8038"},
	{"gallon", "nanoliter", "4546090000"},
	{"gallon", "quart", "4"},
	{"gallon", "fluid pint", "9.607599"},
	{"gallon", "customary fluid ounce", "153.72159"},
	{"gallon", "pint", "8"},
	{"gallon", "microliter", "4546090"},
	{"gallon", "deciliter", "45.4609"},
	{"gallon", "gigaliter", "0.00000000454609"},
	{"gallon", "fluid ounce", "160"},
	{"hectoliter", "femtoliter", "100000000000000000"},
	{"hectoliter", "megaliter", "0.0001"},
	{"hectoliter", "petaliter", "0.0000000000001"},
	{"hectoliter", "teraliter", "0.0000000001"},
	{"hectoliter", "decaliter", "10"},
	{"hectoliter", "kiloliter", "0.1"},
	{"hectoliter", "attoliter", "100000000000000000000"},
	{"hectoliter", "centiliter", "10000"},
	{"hectoliter", "liter", "100"},
	{"hectoliter", "exaliter", "0.0000000000000001"},
	{"hectoliter", "fluid gallon", "26.417205"},
	{"hectoliter", "picoliter", "100000000000000"},
	{"hectoliter", "milliliter", "100000"},
	{"hectoliter", "gallon", "21.996925"},
	{"hectoliter", "fluid quart", "105.668821"},
	{"hectoliter", "nanoliter", "100000000000"},
	{"hectoliter", "quart", "87.987699"},
	{"hectoliter", "fluid pint", "211.337642"},
	{"hectoliter", "customary fluid ounce", "3381.40227"},
	{"hectoliter", "pint", "175.975399"},
	{"hectoliter", "microliter", "100000000"},
	{"hectoliter", "deciliter", "1000"},
	{"hectoliter", "gigaliter", "0.0000001"},
	{"hectoliter", "fluid ounce", "3519.507973"},
	{"fluid quart", "femtoliter", "946352945999999.875"},
	{"fluid quart", "megaliter", "0.000000946353"},
	{"fluid quart", "petaliter", "0.000000000000000946353"},
	{"fluid quart", "teraliter", "0.000000000000946353"},
	{"fluid quart", "decaliter", "0.0946353"},
	{"fluid quart", "kiloliter", "0.000946353"},
	{"fluid quart", "attoliter", "946352945999999872"},
	{"fluid quart", "centiliter", "94.635295"},
	{"fluid quart", "liter", "0.946353"},
	{"fluid quart", "exaliter", "0.000000000000000000946353"},
	{"fluid quart", "fluid gallon", "0.25"},
	{"fluid quart", "picoliter", "946352946000"},
	{"fluid quart", "milliliter", "946.352946"},
	{"fluid quart", "gallon", "0.2081685"},
	{"fluid quart", "hectoliter", "0.00946353"},
	{"fluid quart", "nanoliter", "946352946"},
	{"fluid quart", "quart", "0.832674"},
	{"fluid quart", "fluid pint", "2"},
	{"fluid quart", "customary fluid ounce", "32"},
	{"fluid quart", "pint", "1.665348"},
	{"fluid quart", "microliter", "946352.946"},
	{"fluid quart", "deciliter", "9.463529"},
	{"fluid quart", "gigaliter", "0.000000000946353"},
	{"fluid quart", "fluid ounce", "33.306967"},
	{"nanoliter", "femtoliter", "1000000"},
	{"nanoliter", "megaliter", "0.000000000000001"},
	{"nanoliter", "petaliter", "0.000000000000000000000001"},
	{"nanoliter", "teraliter", "0.000000000000000000001"},
	{"nanoliter", "decaliter", "0.0000000001"},
	{"nanoliter", "kiloliter", "0.000000000001"},
	{"nanoliter", "attoliter", "1000000000"},
	{"nanoliter", "centiliter", "0.0000001"},
	{"nanoliter", "liter", "0.000000001"},
	{"nanoliter", "exaliter", "0.000000000000000000000000001"},
	{"nanoliter", "fluid gallon", "0.0000000002641721"},
	{"nanoliter", "picoliter", "1000"},
	{"nanoliter", "milliliter", "0.000001"},
	{"nanoliter", "gallon", "0.0000000002199692"},
	{"nanoliter", "hectoliter", "0.00000000001"},
	{"nanoliter", "fluid quart", "0.000000001056688"},
	{"nanoliter", "quart", "0.000000000879877"},
	{"nanoliter", "fluid pint", "0.000000002113376"},
	{"nanoliter", "customary fluid ounce", "0.000000033814"},
	{"nanoliter", "pint", "0.000000001759754"},
	{"nanoliter", "microliter", "0.001"},
	{"nanoliter", "deciliter", "0.00000001"},
	{"nanoliter", "gigaliter", "0.000000000000000001"},
	{"nanoliter", "fluid ounce", "0.0000000351951"},
	{"quart", "femtoliter", "1136522500000000"},
	{"quart", "megaliter", "0.000001136523"},
	{"quart", "petaliter", "0.000000000000001136523"},
	{"quart", "teraliter", "0.000000000001136523"},
	{"quart", "decaliter", "0.1136523"},
	{"quart", "kiloliter", "0.001136523"},
	{"quart", "attoliter", "1136522500000000000"},
	{"quart", "centiliter", "113.65225"},
	{"quart", "liter", "1.136523"},
	{"quart", "exaliter", "0.000000000000000001136523"},
	{"quart", "fluid gallon", "0.3002375"},
	{"quart", "picoliter", "1136522500000.000244"},
	{"quart", "milliliter", "1136.5225"},
	{"quart", "gallon", "0.25"},
	{"quart", "hectoliter", "0.01136523"},
	{"quart", "fluid quart", "1.20095"},
	{"quart", "nanoliter", "1136522500"},
	{"quart", "fluid pint", "2.4019"},
	{"quart", "customary fluid ounce", "38.430398"},
	{"quart", "pint", "2"},
	{"quart", "microliter", "1136522.5"},
	{"quart", "deciliter", "11.365225"},
	{"quart", "gigaliter", "0.000000001136523"},
	{"quart", "fluid ounce", "40"},
	{"fluid pint", "femtoliter", "473176472999999.9375"},
	{"fluid pint", "megaliter", "0.000000473176"},
	{"fluid pint", "petaliter", "0.000000000000000473176"},
	{"fluid pint", "teraliter", "0.000000000000473176"},
	{"fluid pint", "decaliter", "0.0473176"},
	{"fluid pint", "kiloliter", "0.000473176"},
	{"fluid pint", "attoliter", "473176472999999936"},
	{"fluid pint", "centiliter", "47.317647"},
	{"fluid pint", "liter", "0.473176"},
	{"fluid pint", "exaliter", "0.000000000000000000473176"},
	{"fluid pint", "fluid gallon", "0.125"},
	{"fluid pint", "picoliter", "473176473000"},
	{"fluid pint", "milliliter", "473.176473"},
	{"fluid pint", "gallon", "0.1040843"},
	{"fluid pint", "hectoliter", "0.00473176"},
	{"fluid pint", "fluid quart", "0.5"},
	{"fluid pint", "nanoliter", "473176473"},
	{"fluid pint", "quart", "0.416337"},
	{"fluid pint", "customary fluid ounce", "16"},
	{"fluid pint", "pint", "0.832674"},
	{"fluid pint", "microliter", "473176.473"},
	{"fluid pint", "deciliter", "4.731765"},
	{"fluid pint", "gigaliter", "0.000000000473176"},
	{"fluid pint", "fluid ounce", "16.653484"},
	{"customary fluid ounce", "femtoliter", "29573529562499.996094"},
	{"customary fluid ounce", "megaliter", "0.00000002957353"},
	{"customary fluid ounce", "petaliter", "0.00000000000000002957353"},
	{"customary fluid ounce", "teraliter", "0.00000000000002957353"},
	{"customary fluid ounce", "decaliter", "0.002957353"},
	{"customary fluid ounce", "kiloliter", "0.00002957353"},
	{"customary fluid ounce", "attoliter", "29573529562499996"},
	{"customary fluid ounce", "centiliter", "2.957353"},
	{"customary fluid ounce", "liter", "0.02957353"},
	{"customary fluid ounce", "exaliter", "0.00000000000000000002957353"},
	{"customary fluid ounce", "fluid gallon", "0.0078125"},
	{"customary fluid ounce", "picoliter", "29573529562.5"},
	{"customary fluid ounce", "milliliter", "29.57353"},
	{"customary fluid ounce", "gallon", "0.00650527"},
	{"customary fluid ounce", "hectoliter", "0.0002957353"},
	{"customary fluid ounce", "fluid quart", "0.03125"},
	{"customary fluid ounce", "nanoliter", "29573529.5625"},
	{"customary fluid ounce", "quart", "0.02602107"},
	{"customary fluid ounce", "fluid pint", "0.0625"},
	{"customary fluid ounce", "pint", "0.0520421"},
	{"customary fluid ounce", "microliter", "29573.529562"},
	{"customary fluid ounce", "deciliter", "0.2957353"},
	{"customary fluid ounce", "gigaliter", "0.00000000002957353"},
	{"customary fluid ounce", "fluid ounce", "1.040843"},
	{"pint", "femtoliter", "568261250000000"},
	{"pint", "megaliter", "0.000000568261"},
	{"pint", "petaliter", "0.000000000000000568261"},
	{"pint", "teraliter", "0.000000000000568261"},
	{"pint", "decaliter", "0.0568261"},
	{"pint", "kiloliter", "0.000568261"},
	{"pint", "attoliter", "568261250000000000"},
	{"pint", "centiliter", "56.826125"},
	{"pint", "liter", "0.568261"},
	{"pint", "exaliter", "0.000000000000000000568261"},
	{"pint", "fluid gallon", "0.1501187"},
	{"pint", "picoliter", "568261250000.000122"},
	{"pint", "milliliter", "568.26125"},
	{"pint", "gallon", "0.125"},
	{"pint", "hectoliter", "0.00568261"},
	{"pint", "fluid quart", "0.600475"},
	{"pint", "nanoliter", "568261250"},
	{"pint", "quart", "0.5"},
	{"pint", "fluid pint", "1.20095"},
	{"pint", "customary fluid ounce", "19.215199"},
	{"pint", "microliter", "568261.25"},
	{"pint", "deciliter", "5.682613"},
	{"pint", "gigaliter", "0.000000000568261"},
	{"pint", "fluid ounce", "20"},
	{"microliter", "femtoliter", "1000000000"},
	{"microliter", "megaliter", "0.000000000001"},
	{"microliter", "petaliter", "0.000000000000000000001"},
	{"microliter", "teraliter", "0.000000000000000001"},
	{"microliter", "decaliter", "0.0000001"},
	{"microliter", "kiloliter", "0.000000001"},
	{"microliter", "attoliter", "999999999999.999878"},
	{"microliter", "centiliter", "0.0001"},
	{"microliter", "liter", "0.000001"},
	{"microliter", "exaliter", "0.000000000000000000000001"},
	{"microliter", "fluid gallon", "0.0000002641721"},
	{"microliter", "picoliter", "1000000"},
	{"microliter", "milliliter", "0.001"},
	{"microliter", "gallon", "0.0000002199692"},
	{"microliter", "hectoliter", "0.00000001"},
	{"microliter", "fluid quart", "0.000001056688"},
	{"microliter", "nanoliter", "1000"},
	{"microliter", "quart", "0.000000879877"},
	{"microliter", "fluid pint", "0.000002113376"},
	{"microliter", "customary fluid ounce", "0.000033814"},
	{"microliter", "pint", "0.000001759754"},
	{"microliter", "deciliter", "0.00001"},
	{"microliter", "gigaliter", "0.000000000000001"},
	{"microliter", "fluid ounce", "0.0000351951"},
	{"deciliter", "femtoliter", "100000000000000"},
	{"deciliter", "megaliter", "0.0000001"},
	{"deciliter", "petaliter", "0.0000000000000001"},
	{"deciliter", "teraliter", "0.0000000000001"},
	{"deciliter", "decaliter", "0.01"},
	{"deciliter", "kiloliter", "0.0001"},
	{"deciliter", "attoliter", "100000000000000000"},
	{"deciliter", "centiliter", "10"},
	{"deciliter", "liter", "0.1"},
	{"deciliter", "exaliter", "0.0000000000000000001"},
	{"deciliter", "fluid gallon", "0.02641721"},
	{"deciliter", "picoliter", "100000000000"},
	{"deciliter", "milliliter", "100"},
	{"deciliter", "gallon", "0.02199692"},
	{"deciliter", "hectoliter", "0.001"},
	{"deciliter", "fluid quart", "0.1056688"},
	{"deciliter", "nanoliter", "100000000"},
	{"deciliter", "quart", "0.0879877"},
	{"deciliter", "fluid pint", "0.2113376"},
	{"deciliter", "customary fluid ounce", "3.381402"},
	{"deciliter", "pint", "0.1759754"},
	{"deciliter", "microliter", "100000"},
	{"deciliter", "gigaliter", "0.0000000001"},
	{"deciliter", "fluid ounce", "3.519508"},
	{"gigaliter", "femtoliter", "999999999999999983222784"},
	{"gigaliter", "megaliter", "1000"},
	{"gigaliter", "petaliter", "0.000001"},
	{"gigaliter", "teraliter", "0.001"},
	{"gigaliter", "decaliter", "100000000"},
	{"gigaliter", "kiloliter", "1000000"},
	{"gigaliter", "attoliter", "999999999999999875848601600"},
	{"gigaliter", "centiliter", "100000000000"},
	{"gigaliter", "liter", "1000000000"},
	{"gigaliter", "exaliter", "0.000000001"},
	{"gigaliter", "fluid gallon", "264172052.358148"},
	{"gigaliter", "picoliter", "1000000000000000000000"},
	{"gigaliter", "milliliter", "1000000000000"},
	{"gigaliter", "gallon", "219969248.299088"},
	{"gigaliter", "hectoliter", "10000000"},
	{"gigaliter", "fluid quart", "1056688209.432594"},
	{"gigaliter", "nanoliter", "1000000000000000000"},
	{"gigaliter", "quart", "879876993.196351"},
	{"gigaliter", "fluid pint", "2113376418.865187"},
	{"gigaliter", "customary fluid ounce", "33814022701.842999"},
	{"gigaliter", "pint", "1759753986.392702"},
	{"gigaliter", "microliter", "1000000000000000"},
	{"gigaliter", "deciliter", "10000000000"},
	{"gigaliter", "fluid ounce", "35195079727.85405"},
	{"fluid ounce", "femtoliter", "28413062499999.996094"},
	{"fluid ounce", "megaliter", "0.00000002841306"},
	{"fluid ounce", "petaliter", "0.00000000000000002841306"},
	{"fluid ounce", "teraliter", "0.00000000000002841306"},
	{"fluid ounce", "decaliter", "0.002841306"},
	{"fluid ounce", "kiloliter", "0.00002841306"},
	{"fluid ounce", "attoliter", "28413062499999996"},
	{"fluid ounce", "centiliter", "2.841306"},
	{"fluid ounce", "liter", "0.02841306"},
	{"fluid ounce", "exaliter", "0.00000000000000000002841306"},
	{"fluid ounce", "fluid gallon", "0.00750594"},
	{"fluid ounce", "picoliter", "28413062500"},
	{"fluid ounce", "milliliter", "28.413062"},
	{"fluid ounce", "gallon", "0.00625"},
	{"fluid ounce", "hectoliter", "0.0002841306"},
	{"fluid ounce", "fluid quart", "0.03002375"},
	{"fluid ounce", "nanoliter", "28413062.5"},
	{"fluid ounce", "quart", "0.025"},
	{"fluid ounce", "fluid pint", "0.0600475"},
	{"fluid ounce", "customary fluid ounce", "0.96076"},
	{"fluid ounce", "pint", "0.05"},
	{"fluid ounce", "microliter", "28413.0625"},
	{"fluid ounce", "deciliter", "0.2841306"},
	{"fluid ounce", "gigaliter", "0.00000000002841306"},
	{"hectogram", "drachm", "56.438339"},
	{"hectogram", "decigram", "1000"},
	{"hectogram", "milligram", "100000"},
	{"hectogram", "pound", "0.2204623"},
	{"hectogram", "grain", "1543.235835"},
	{"hectogram", "petagram", "0.0000000000001"},
	{"hectogram", "gigagram", "0.0000001"},
	{"hectogram", "centigram", "10000"},
	{"hectogram", "ton", "0.0000984207"},
	{"hectogram", "teragram", "0.0000000001"},
	{"hectogram", "nanogram", "100000000000"},
	{"hectogram", "picogram", "100000000000000"},
	{"hectogram", "microgram", "100000000"},
	{"hectogram", "ounce", "3.527396"},
	{"hectogram", "exagram", "0.0000000000000001"},
	{"hectogram", "attogram", "100000000000000000000"},
	{"hectogram", "kilogram", "0.1"},
	{"hectogram", "megagram", "0.0001"},
	{"hectogram", "stone", "0.0157473"},
	{"hectogram", "femtogram", "100000000000000000"},
	{"hectogram", "gram", "100"},
	{"hectogram", "slug", "0.00685218"},
	{"hectogram", "decagram", "10"},
	{"drachm", "hectogram", "0.01771845"},
	{"drachm", "decigram", "17.718452"},
	{"drachm", "milligram", "1771.845195"},
	{"drachm", "pound", "0.00390625"},
	{"drachm", "grain", "27.34375"},
	{"drachm", "petagram", "0.000000000000001771845"},
	{"drachm", "gigagram", "0.000000001771845"},
	{"drachm", "centigram", "177.18452"},
	{"drachm", "ton", "0.000001743862"},
	{"drachm", "teragram", "0.000000000001771845"},
	{"drachm", "nanogram", "1771845195.3125"},
	{"drachm", "picogram", "1771845195312.5"},
	{"drachm", "microgram", "1771845.195313"},
	{"drachm", "ounce", "0.0625"},
	{"drachm", "exagram", "0.000000000000000001771845"},
	{"drachm", "attogram", "1771845195312499968"},
	{"drachm", "kilogram", "0.001771845"},
	{"drachm", "megagram", "0.000001771845"},
	{"drachm", "stone", "0.0002790179"},
	{"drachm", "femtogram", "1771845195312500"},
	{"drachm", "gram", "1.771845"},
	{"drachm", "slug", "0.00012141"},
	{"drachm", "decagram", "0.1771845"},
	{"decigram", "hectogram", "0.001"},
	{"decigram", "drachm", "0.0564383"},
	{"decigram", "milligram", "100"},
	{"decigram", "pound", "0.0002204623"},
	{"decigram", "grain", "1.543236"},
	{"decigram", "petagram", "0.0000000000000001"},
	{"decigram", "gigagram", "0.0000000001"},
	{"decigram", "centigram", "10"},
	{"decigram", "ton", "0.0000000984207"},
	{"decigram", "teragram", "0.0000000000001"},
	{"decigram", "nanogram", "100000000"},
	{"decigram", "picogram", "100000000000"},
	{"decigram", "microgram", "100000"},
	{"decigram", "ounce", "0.0035274"},
	{"decigram", "exagram", "0.0000000000000000001"},
	{"decigram", "attogram", "100000000000000000"},
	{"decigram", "kilogram", "0.0001"},
	{"decigram", "megagram", "0.0000001"},
	{"decigram", "stone", "0.0000157473"},
	{"decigram", "femtogram", "100000000000000"},
	{"decigram", "gram", "0.1"},
	{"decigram", "slug", "0.00000685218"},
	{"decigram", "decagram", "0.01"},
	{"milligram", "hectogram", "0.00001"},
	{"milligram", "drachm", "0.000564383"},
	{"milligram", "decigram", "0.01"},
	{"milligram", "pound", "0.000002204623"},
	{"milligram", "grain", "0.01543236"},
	{"milligram", "petagram", "0.000000000000000001"},
	{"milligram", "gigagram", "0.000000000001"},
	{"milligram", "centigram", "0.1"},
	{"milligram", "ton", "0.000000000984207"},
	{"milligram", "teragram", "0.000000000000001"},
	{"milligram", "nanogram", "1000000"},
	{"milligram", "picogram", "1000000000"},
	{"milligram", "microgram", "1000"},
	{"milligram", "ounce", "0.000035274"},
	{"milligram", "exagram", "0.000000000000000000001"},
	{"milligram", "attogram", "1000000000000000"},
	{"milligram", "kilogram", "0.000001"},
	{"milligram", "megagram", "0.000000001"},
	{"milligram", "stone", "0.000000157473"},
	{"milligram", "femtogram", "1000000000000"},
	{"milligram", "gram", "0.001"},
	{"milligram", "slug", "0.0000000685218"},
	{"milligram", "decagram", "0.0001"},
	{"pound", "hectogram", "4.535924"},
	{"pound", "drachm", "256"},
	{"pound", "decigram", "4535.9237"},
	{"pound", "milligram", "453592.37"},
	{"pound", "grain", "7000"},
	{"pound", "petagram", "0.000000000000453592"},
	{"pound", "gigagram", "0.000000453592"},
	{"pound", "centigram", "45359.237"},
	{"pound", "ton", "0.000446429"},
	{"pound", "teragram", "0.000000000453592"},
	{"pound", "nanogram", "453592370000"},
	{"pound", "picogram", "453592370000000"},
	{"pound", "microgram", "453592370"},
	{"pound", "ounce", "16"},
	{"pound", "exagram", "0.000000000000000453592"},
	{"pound", "attogram", "453592369999999991808"},
	{"pound", "kilogram", "0.453592"},
	{"pound", "megagram", "0.000453592"},
	{"pound", "stone", "0.0714286"},
	{"pound", "femtogram", "453592370000000000"},
	{"pound", "gram", "453.59237"},
	{"pound", "slug", "0.03108095"},
	{"pound", "decagram", "45.359237"},
	{"grain", "hectogram", "0.000647989"},
	{"grain", "drachm", "0.0365714"},
	{"grain", "decigram", "0.647989"},
	{"grain", "milligram", "64.79891"},
	{"grain", "pound", "0.0001428571"},
	{"grain", "petagram", "0.0000000000000000647989"},
	{"grain", "gigagram", "0.0000000000647989"},
	{"grain", "centigram", "6.479891"},
	{"grain", "ton", "0.0000000637755"},
	{"grain", "teragram", "0.0000000000000647989"},
	{"grain", "nanogram", "64798910"},
	{"grain", "picogram", "64798910000"},
	{"grain", "microgram", "64798.91"},
	{"grain", "ounce", "0.002285714"},
	{"grain", "exagram", "0.0000000000000000000647989"},
	{"grain", "attogram", "64798910000000000"},
	{"grain", "kilogram", "0.0000647989"},
	{"grain", "megagram", "0.0000000647989"},
	{"grain", "stone", "0.00001020408"},
	{"grain", "femtogram", "64798910000000"},
	{"grain", "gram", "0.0647989"},
	{"grain", "slug", "0.00000444014"},
	{"grain", "decagram", "0.00647989"},
	{"petagram", "hectogram", "10000000000000"},
	{"petagram", "drachm", "564383391193286.625"},
	{"petagram", "decigram", "10000000000000000"},
	{"petagram", "milligram", "1000000000000000000"},
	{"petagram", "pound", "2204622621848.775879"},
	{"petagram", "grain", "15432358352941430"},
	{"petagram", "gigagram", "1000000"},
	{"petagram", "centigram", "100000000000000000"},
	{"petagram", "ton", "984206527.611061"},
	{"petagram", "teragram", "1000"},
	{"petagram", "nanogram", "999999999999999983222784"},
	{"petagram", "picogram", "1000000000000000013287555072"},
	{"petagram", "microgram", "1000000000000000000000"},
	{"petagram", "ounce", "35273961949580.414062"},
	{"petagram", "exagram", "0.001"},
	{"petagram", "attogram", "999999999999999945575230987042816"},
	{"petagram", "kilogram", "1000000000000"},
	{"petagram", "megagram", "1000000000"},
	{"petagram", "stone", "157473044417.769714"},
	{"petagram", "femtogram", "999999999999999879147136483328"},
	{"petagram", "gram", "1000000000000000"},
	{"petagram", "slug", "68521765843.674988"},
	{"petagram", "decagram", "100000000000000"},
	{"gigagram", "hectogram", "10000000"},
	{"gigagram", "drachm", "564383391.193287"},
	{"gigagram", "decigram", "10000000000"},
	{"gigagram", "milligram", "1000000000000"},
	{"gigagram", "pound", "2204622.621849"},
	{"gigagram", "grain", "15432358352.941431"},
	{"gigagram", "petagram", "0.000001"},
	{"gigagram", "centigram", "100000000000"},
	{"gigagram", "ton", "984.206528"},
	{"gigagram", "teragram", "0.001"},
	{"gigagram", "nanogram", "1000000000000000000"},
	{"gigagram", "picogram", "1000000000000000000000"},
	{"gigagram", "microgram", "1000000000000000"},
	{"gigagram", "ounce", "35273961.94958"},
	{"gigagram", "exagram", "0.000000001"},
	{"gigagram", "attogram", "999999999999999875848601600"},
	{"gigagram", "kilogram", "1000000"},
	{"gigagram", "megagram", "1000"},
	{"gigagram", "stone", "157473.044418"},
	{"gigagram", "femtogram", "999999999999999983222784"},
	{"gigagram", "gram", "1000000000"},
	{"gigagram", "slug", "68521.765844"},
	{"gigagram", "decagram", "100000000"},
	{"centigram", "hectogram", "0.0001"},
	{"centigram", "drachm", "0.00564383"},
	{"centigram", "decigram", "0.1"},
	{"centigram", "milligram", "10"},
	{"centigram", "pound", "0.00002204623"},
	{"centigram", "grain", "0.1543236"},
	{"centigram", "petagram", "0.00000000000000001"},
	{"centigram", "gigagram", "0.00000000001"},
	{"centigram", "ton", "0.00000000984207"},
	{"centigram", "teragram", "0.00000000000001"},
	{"centigram", "nanogram", "10000000"},
	{"centigram", "picogram", "10000000000"},
	{"centigram", "microgram", "10000"},
	{"centigram", "ounce", "0.00035274"},
	{"centigram", "exagram", "0.00000000000000000001"},
	{"centigram", "attogram", "10000000000000000"},
	{"centigram", "kilogram", "0.00001"},
	{"centigram", "megagram", "0.00000001"},
	{"centigram", "stone", "0.00000157473"},
	{"centigram", "femtogram", "10000000000000"},
	{"centigram", "gram", "0.01"},
	{"centigram", "slug", "0.000000685218"},
	{"centigram", "decagram", "0.001"},
	{"ton", "hectogram", "10160.469088"},
	{"ton", "drachm", "573440"},
	{"ton", "decigram", "10160469.088"},
	{"ton", "milligram", "1016046908.8"},
	{"ton", "pound", "2240"},
	{"ton", "grain", "15680000"},
	{"ton", "petagram", "0.000000001016047"},
	{"ton", "gigagram", "0.001016047"},
	{"ton", "centigram", "101604690.88"},
	{"ton", "teragram", "0.000001016047"},
	{"ton", "nanogram", "1016046908799999.875"},
	{"ton", "picogram", "1016046908800000000"},
	{"ton", "microgram", "1016046908800"},
	{"ton", "ounce", "35840"},
	{"ton", "exagram", "0.000000000001016047"},
	{"ton", "attogram", "1016046908799999952289792"},
	{"ton", "kilogram", "1016.046909"},
	{"ton", "megagram", "1.016047"},
	{"ton", "stone", "160"},
	{"ton", "femtogram", "1016046908799999934464"},
	{"ton", "gram", "1016046.9088"},
	{"ton", "slug", "69.621328"},
	{"ton", "decagram", "101604.69088"},
	{"teragram", "hectogram", "10000000000"},
	{"teragram", "drachm", "564383391193.286621"},
	{"teragram", "decigram", "10000000000000"},
	{"teragram", "milligram", "1000000000000000"},
	{"teragram", "pound", "2204622621.848776"},
	{"teragram", "grain", "15432358352941.429688"},
	{"teragram", "petagram", "0.001"},
	{"teragram", "gigagram", "1000"},
	{"teragram", "centigram", "100000000000000"},
	{"teragram", "ton", "984206.527611"},
	{"teragram", "nanogram", "1000000000000000000000"},
	{"teragram", "picogram", "999999999999999983222784"},
	{"teragram", "microgram", "1000000000000000000"},
	{"teragram", "ounce", "35273961949.580414"},
	{"teragram", "exagram", "0.000001"},
	{"teragram", "attogram", "999999999999999879147136483328"},
	{"teragram", "kilogram", "1000000000"},
	{"teragram", "megagram", "1000000"},
	{"teragram", "stone", "157473044.41777"},
	{"teragram", "femtogram", "999999999999999875848601600"},
	{"teragram", "gram", "1000000000000"},
	{"teragram", "slug", "68521765.843675"},
	{"teragram", "decagram", "100000000000"},
	{"nanogram", "hectogram", "0.00000000001"},
	{"nanogram", "drachm", "0.000000000564383"},
	{"nanogram", "decigram", "0.00000001"},
	{"nanogram", "milligram", "0.000001"},
	{"nanogram", "pound", "0.000000000002204623"},
	{"nanogram", "grain", "0.00000001543236"},
	{"nanogram", "petagram", "0.000000000000000000000001"},
	{"nanogram", "gigagram", "0.000000000000000001"},
	{"nanogram", "centigram", "0.0000001"},
	{"nanogram", "ton", "0.000000000000000984207"},
	{"nanogram", "teragram", "0.000000000000000000001"},
	{"nanogram", "picogram", "1000"},
	{"nanogram", "microgram", "0.001"},
	{"nanogram", "ounce", "0.000000000035274"},
	{"nanogram", "exagram", "0.000000000000000000000000001"},
	{"nanogram", "attogram", "1000000000"},
	{"nanogram", "kilogram", "0.000000000001"},
	{"nanogram", "megagram", "0.000000000000001"},
	{"nanogram", "stone", "0.000000000000157473"},
	{"nanogram", "femtogram", "1000000"},
	{"nanogram", "gram", "0.000000001"},
	{"nanogram", "slug", "0.0000000000000685218"},
	{"nanogram", "decagram", "0.0000000001"},
	{"picogram", "hectogram", "0.00000000000001"},
	{"picogram", "drachm", "0.000000000000564383"},
	{"picogram", "decigram", "0.00000000001"},
	{"picogram", "milligram", "0.000000001"},
	{"picogram", "pound", "0.000000000000002204623"},
	{"picogram", "grain", "0.00000000001543236"},
	{"picogram", "petagram", "0.000000000000000000000000001"},
	{"picogram", "gigagram", "0.000000000000000000001"},
	{"picogram", "centigram", "0.0000000001"},
	{"picogram", "ton", "0.000000000000000000984207"},
	{"picogram", "teragram", "0.000000000000000000000001"},
	{"picogram", "nanogram", "0.001"},
	{"picogram", "microgram", "0.000001"},
	{"picogram", "ounce", "0.000000000000035274"},
	{"picogram", "exagram", "0.000000000000000000000000000001"},
	{"picogram", "attogram", "1000000"},
	{"picogram", "kilogram", "0.000000000000001"},
	{"picogram", "megagram", "0.000000000000000001"},
	{"picogram", "stone", "0.000000000000000157473"},
	{"picogram", "femtogram", "1000"},
	{"picogram", "gram", "0.000000000001"},
	{"picogram", "slug", "0.0000000000000000685218"},
	{"picogram", "decagram", "0.0000000000001"},
	{"microgram", "hectogram", "0.00000001"},
	{"microgram", "drachm", "0.000000564383"},
	{"microgram", "decigram", "0.00001"},
	{"microgram", "milligram", "0.001"},
	{"microgram", "pound", "0.000000002204623"},
	{"microgram", "grain", "0.00001543236"},
	{"microgram", "petagram", "0.000000000000000000001"},
	{"microgram", "gigagram", "0.000000000000001"},
	{"microgram", "centigram", "0.0001"},
	{"microgram", "ton", "0.000000000000984207"},
	{"microgram", "teragram", "0.000000000000000001"},
	{"microgram", "nanogram", "1000"},
	{"microgram", "picogram", "1000000"},
	{"microgram", "ounce", "0.000000035274"},
	{"microgram", "exagram", "0.000000000000000000000001"},
	{"microgram", "attogram", "999999999999.999878"},
	{"microgram", "kilogram", "0.000000001"},
	{"microgram", "megagram", "0.000000000001"},
	{"microgram", "stone", "0.000000000157473"},
	{"microgram", "femtogram", "1000000000"},
	{"microgram", "gram", "0.000001"},
	{"microgram", "slug", "0.0000000000685218"},
	{"microgram", "decagram", "0.0000001"},
	{"ounce", "hectogram", "0.2834952"},
	{"ounce", "drachm", "16"},
	{"ounce", "decigram", "283.495231"},
	{"ounce", "milligram", "28349.523125"},
	{"ounce", "pound", "0.0625"},
	{"ounce", "grain", "437.5"},
	{"ounce", "petagram", "0.00000000000002834952"},
	{"ounce", "gigagram", "0.00000002834952"},
	{"ounce", "centigram", "2834.952313"},
	{"ounce", "ton", "0.00002790179"},
	{"ounce", "teragram", "0.00000000002834952"},
	{"ounce", "nanogram", "28349523125"},
	{"ounce", "picogram", "28349523125000"},
	{"ounce", "microgram", "28349523.125"},
	{"ounce", "exagram", "0.00000000000000002834952"},
	{"ounce", "attogram", "28349523124999999488"},
	{"ounce", "kilogram", "0.02834952"},
	{"ounce", "megagram", "0.00002834952"},
	{"ounce", "stone", "0.00446429"},
	{"ounce", "femtogram", "28349523125000000"},
	{"ounce", "gram", "28.349523"},
	{"ounce", "slug", "0.001942559"},
	{"ounce", "decagram", "2.834952"},
	{"exagram", "hectogram", "10000000000000000"},
	{"exagram", "drachm", "564383391193286592"},
	{"exagram", "decigram", "10000000000000000000"},
	{"exagram", "milligram", "1000000000000000000000"},
	{"exagram", "pound", "2204622621848775.75"},
	{"exagram", "grain", "15432358352941430784"},
	{"exagram", "petagram", "1000"},
	{"exagram", "gigagram", "1000000000"},
	{"exagram", "centigram", "100000000000000000000"},
	{"exagram", "ton", "984206527611.060669"},
	{"exagram", "teragram", "1000000"},
	{"exagram", "nanogram", "999999999999999875848601600"},
	{"exagram", "picogram", "1000000000000000019884624838656"},
	{"exagram", "microgram", "999999999999999983222784"},
	{"exagram", "ounce", "35273961949580412"},
	{"exagram", "attogram", "999999999999999894846684784341549056"},
	{"exagram", "kilogram", "1000000000000000"},
	{"exagram", "megagram", "1000000000000"},
	{"exagram", "stone", "157473044417769.71875"},
	{"exagram", "femtogram", "999999999999999945575230987042816"},
	{"exagram", "gram", "1000000000000000000"},
	{"exagram", "slug", "68521765843674.992188"},
	{"exagram", "decagram", "100000000000000000"},
	{"attogram", "hectogram", "0.00000000000000000001"},
	{"attogram", "drachm", "0.000000000000000000564383"},
	{"attogram", "decigram", "0.00000000000000001"},
	{"attogram", "milligram", "0.000000000000001"},
	{"attogram", "pound", "0.000000000000000000002204623"},
	{"attogram", "grain", "0.00000000000000001543236"},
	{"attogram", "petagram", "0.000000000000000000000000000000001"},
	{"attogram", "gigagram", "0.000000000000000000000000001"},
	{"attogram", "centigram", "0.0000000000000001"},
	{"attogram", "ton", "0.000000000000000000000000984207"},
	{"attogram", "teragram", "0.000000000000000000000000000001"},
	{"attogram", "nanogram", "0.000000001"},
	{"attogram", "picogram", "0.000001"},
	{"attogram", "microgram", "0.000000000001"},
	{"attogram", "ounce", "0.000000000000000000035274"},
	{"attogram", "exagram", "0.000000000000000000000000000000000001"},
	{"attogram", "kilogram", "0.000000000000000000001"},
	{"attogram", "megagram", "0.000000000000000000000001"},
	{"attogram", "stone", "0.000000000000000000000157473"},
	{"attogram", "femtogram", "0.001"},
	{"attogram", "gram", "0.000000000000000001"},
	{"attogram", "slug", "0.0000000000000000000000685218"},
	{"attogram", "decagram", "0.0000000000000000001"},
	{"kilogram", "hectogram", "10"},
	{"kilogram", "drachm", "564.383391"},
	{"kilogram", "decigram", "10000"},
	{"kilogram", "milligram", "1000000"},
	{"kilogram", "pound", "2.204623"},
	{"kilogram", "grain", "15432.358353"},
	{"kilogram", "petagram", "0.000000000001"},
	{"kilogram", "gigagram", "0.000001"},
	{"kilogram", "centigram", "100000"},
	{"kilogram", "ton", "0.000984207"},
	{"kilogram", "teragram", "0.000000001"},
	{"kilogram", "nanogram", "999999999999.999878"},
	{"kilogram", "picogram", "1000000000000000"},
	{"kilogram", "microgram", "1000000000"},
	{"kilogram", "ounce", "35.273962"},
	{"kilogram", "exagram", "0.000000000000001"},
	{"kilogram", "attogram", "999999999999999868928"},
	{"kilogram", "megagram", "0.001"},
	{"kilogram", "stone", "0.157473"},
	{"kilogram", "femtogram", "999999999999999872"},
	{"kilogram", "gram", "1000"},
	{"kilogram", "slug", "0.0685218"},
	{"kilogram", "decagram", "100"},
	{"megagram", "hectogram", "10000"},
	{"megagram", "drachm", "564383.391193"},
	{"megagram", "decigram", "10000000"},
	{"megagram", "milligram", "1000000000"},
	{"megagram", "pound", "2204.622622"},
	{"megagram", "grain", "15432358.352941"},
	{"megagram", "petagram", "0.000000001"},
	{"megagram", "gigagram", "0.001"},
	{"megagram", "centigram", "100000000"},
	{"megagram", "ton", "0.984207"},
	{"megagram", "teragram", "0.000001"},
	{"megagram", "nanogram", "1000000000000000"},
	{"megagram", "picogram", "1000000000000000000"},
	{"megagram", "microgram", "1000000000000"},
	{"megagram", "ounce", "35273.96195"},
	{"megagram", "exagram", "0.000000000001"},
	{"megagram", "attogram", "999999999999999983222784"},
	{"megagram", "kilogram", "1000"},
	{"megagram", "stone", "157.473044"},
	{"megagram", "femtogram", "999999999999999868928"},
	{"megagram", "gram", "1000000"},
	{"megagram", "slug", "68.521766"},
	{"megagram", "decagram", "100000"},
	{"stone", "hectogram", "63.502932"},
	{"stone", "drachm", "3584"},
	{"stone", "decigram", "63502.9318"},
	{"stone", "milligram", "6350293.18"},
	{"stone", "pound", "14"},
	{"stone", "grain", "98000"},
	{"stone", "petagram", "0.00000000000635029"},
	{"stone", "gigagram", "0.00000635029"},
	{"stone", "centigram", "635029.318"},
	{"stone", "ton", "0.00625"},
	{"stone", "teragram", "0.00000000635029"},
	{"stone", "nanogram", "6350293179999.999023"},
	{"stone", "picogram", "6350293180000000"},
	{"stone", "microgram", "6350293180"},
	{"stone", "ounce", "224"},
	{"stone", "exagram", "0.00000000000000635029"},
	{"stone", "attogram", "6350293179999999492096"},
	{"stone", "kilogram", "6.350293"},
	{"stone", "megagram", "0.00635029"},
	{"stone", "femtogram", "6350293179999998976"},
	{"stone", "gram", "6350.29318"},
	{"stone", "slug", "0.435133"},
	{"stone", "decagram", "635.029318"},
	{"femtogram", "hectogram", "0.00000000000000001"},
	{"femtogram", "drachm", "0.000000000000000564383"},
	{"femtogram", "decigram", "0.00000000000001"},
	{"femtogram", "milligram", "0.000000000001"},
	{"femtogram", "pound", "0.000000000000000002204623"},
	{"femtogram", "grain", "0.00000000000001543236"},
	{"femtogram", "petagram", "0.000000000000000000000000000001"},
	{"femtogram", "gigagram", "0.000000000000000000000001"},
	{"femtogram", "centigram", "0.0000000000001"},
	{"femtogram", "ton", "0.000000000000000000000984207"},
	{"femtogram", "teragram", "0.000000000000000000000000001"},
	{"femtogram", "nanogram", "0.000001"},
	{"femtogram", "picogram", "0.001"},
	{"femtogram", "microgram", "0.000000001"},
	{"femtogram", "ounce", "0.000000000000000035274"},
	{"femtogram", "exagram", "0.000000000000000000000000000000001"},
	{"femtogram", "attogram", "1000"},
	{"femtogram", "kilogram", "0.000000000000000001"},
	{"femtogram", "megagram", "0.000000000000000000001"},
	{"femtogram", "stone", "0.000000000000000000157473"},
	{"femtogram", "gram", "0.000000000000001"},
	{"femtogram", "slug", "0.0000000000000000000685218"},
	{"femtogram", "decagram", "0.0000000000000001"},
	{"gram", "hectogram", "0.01"},
	{"gram", "drachm", "0.564383"},
	{"gram", "decigram", "10"},
	{"gram", "milligram", "1000"},
	{"gram", "pound", "0.002204623"},
	{"gram", "grain", "15.432358"},
	{"gram", "petagram", "0.000000000000001"},
	{"gram", "gigagram", "0.000000001"},
	{"gram", "centigram", "100"},
	{"gram", "ton", "0.000000984207"},
	{"gram", "teragram", "0.000000000001"},
	{"gram", "nanogram", "1000000000"},
	{"gram", "picogram", "1000000000000"},
	{"gram", "microgram", "1000000"},
	{"gram", "ounce", "0.035274"},
	{"gram", "exagram", "0.000000000000000001"},
	{"gram", "attogram", "999999999999999872"},
	{"gram", "kilogram", "0.001"},
	{"gram", "megagram", "0.000001"},
	{"gram", "stone", "0.000157473"},
	{"gram", "femtogram", "999999999999999.875"},
	{"gram", "slug", "0.0000685218"},
	{"gram", "decagram", "0.1"},
	{"slug", "hectogram", "145.939029"},
	{"slug", "drachm", "8236.556432"},
	{"slug", "decigram", "145939.0294"},
	{"slug", "milligram", "14593902.94"},
	{"slug", "pound", "32.174049"},
	{"slug", "grain", "225218.339938"},
	{"slug", "petagram", "0.0000000000145939"},
	{"slug", "gigagram", "0.0000145939"},
	{"slug", "centigram", "1459390.294"},
	{"slug", "ton", "0.01436341"},
	{"slug", "teragram", "0.0000000145939"},
	{"slug", "nanogram", "14593902940000"},
	{"slug", "picogram", "14593902940000000"},
	{"slug", "microgram", "14593902940"},
	{"slug", "ounce", "514.784777"},
	{"slug", "exagram", "0.0000000000000145939"},
	{"slug", "attogram", "14593902939999999885312"},
	{"slug", "kilogram", "14.593903"},
	{"slug", "megagram", "0.0145939"},
	{"slug", "stone", "2.298146"},
	{"slug", "femtogram", "14593902939999997952"},
	{"slug", "gram", "14593.90294"},
	{"slug", "decagram", "1459.390294"},
	{"decagram", "hectogram", "0.1"},
	{"decagram", "drachm", "5.643834"},
	{"decagram", "decigram", "100"},
	{"decagram", "milligram", "10000"},
	{"decagram", "pound", "0.02204623"},
	{"decagram", "grain", "154.323584"},
	{"decagram", "petagram", "0.00000000000001"},
	{"decagram", "gigagram", "0.00000001"},
	{"decagram", "centigram", "1000"},
	{"decagram", "ton", "0.00000984207"},
	{"decagram", "teragram", "0.00000000001"},
	{"decagram", "nanogram", "10000000000"},
	{"decagram", "picogram", "10000000000000"},
	{"decagram", "microgram", "10000000"},
	{"decagram", "ounce", "0.35274"},
	{"decagram", "exagram", "0.00000000000000001"},
	{"decagram", "attogram", "10000000000000000000"},
	{"decagram", "kilogram", "0.01"},
	{"decagram", "megagram", "0.00001"},
	{"decagram", "stone", "0.00157473"},
	{"decagram", "femtogram", "10000000000000000"},
	{"decagram", "gram", "10"},
	{"decagram", "slug", "0.000685218"},
	{"kibibyte", "pebibyte", "0.000000000000909495"},
	{"kibibyte", "yobibyte", "0.000000000000000000000847033"},
	{"kibibyte", "exbibyte", "0.000000000000000888178"},
	{"kibibyte", "gigabit", "0.000008192"},
	{"kibibyte", "bit", "8192"},
	{"kibibyte", "nibble", "2048"},
	{"kibibyte", "megabit", "0.008192"},
	{"kibibyte", "zebibyte", "0.000000000000000000867362"},
	{"kibibyte", "byte", "1024"},
	{"kibibyte", "petabit", "0.000000000008192"},
	{"kibibyte", "mebibyte", "0.000976562"},
	{"kibibyte", "exabit", "0.000000000000008192"},
	{"kibibyte", "terabit", "0.000000008192"},
	{"kibibyte", "Tebibyte", "0.000000000931323"},
	{"kibibyte", "gibibyte", "0.000000953674"},
	{"kibibyte", "kilobit", "8.192"},
	{"pebibyte", "kibibyte", "1099511627776"},
	{"pebibyte", "yobibyte", "0.000000000931323"},
	{"pebibyte", "exbibyte", "0.000976562"},
	{"pebibyte", "gigabit", "9007199.254741"},
	{"pebibyte", "bit", "9007199254740992"},
	{"pebibyte", "nibble", "2251799813685248"},
	{"pebibyte", "megabit", "9007199254.740992"},
	{"pebibyte", "zebibyte", "0.000000953674"},
	{"pebibyte", "byte", "1125899906842624"},
	{"pebibyte", "petabit", "9.007199"},
	{"pebibyte", "mebibyte", "1073741824"},
	{"pebibyte", "exabit", "0.0090072"},
	{"pebibyte", "terabit", "9007.199255"},
	{"pebibyte", "Tebibyte", "1024"},
	{"pebibyte", "gibibyte", "1048576"},
	{"pebibyte", "kilobit", "9007199254740.992188"},
	{"yobibyte", "kibibyte", "1180591620717411303424"},
	{"yobibyte", "pebibyte", "1073741824"},
	{"yobibyte", "exbibyte", "1048576"},
	{"yobibyte", "gigabit", "9671406556917034"},
	{"yobibyte", "bit", "9671406556917033397649408"},
	{"yobibyte", "nibble", "2417851639229258349412352"},
	{"yobibyte", "megabit", "9671406556917032960"},
	{"yobibyte", "zebibyte", "1024"},
	{"yobibyte", "byte", "1208925819614629174706176"},
	{"yobibyte", "petabit", "9671406556.917034"},
	{"yobibyte", "mebibyte", "1152921504606846976"},
	{"yobibyte", "exabit", "9671406.556917"},
	{"yobibyte", "terabit", "9671406556917.033203"},
	{"yobibyte", "Tebibyte", "1099511627776"},
	{"yobibyte", "gibibyte", "1125899906842624"},
	{"yobibyte", "kilobit", "9671406556917033598976"},
	{"exbibyte", "kibibyte", "1125899906842624"},
	{"exbibyte", "pebibyte", "1024"},
	{"exbibyte", "yobibyte", "0.000000953674"},
	{"exbibyte", "gigabit", "9223372036.854776"},
	{"exbibyte", "bit", "9223372036854775808"},
	{"exbibyte", "nibble", "2305843009213693952"},
	{"exbibyte", "megabit", "9223372036854.775391"},
	{"exbibyte", "zebibyte", "0.000976562"},
	{"exbibyte", "byte", "1152921504606846976"},
	{"exbibyte", "petabit", "9223.372037"},
	{"exbibyte", "mebibyte", "1099511627776"},
	{"exbibyte", "exabit", "9.223372"},
	{"exbibyte", "terabit", "9223372.036855"},
	{"exbibyte", "Tebibyte", "1048576"},
	{"exbibyte", "gibibyte", "1073741824"},
	{"exbibyte", "kilobit", "9223372036854776"},
	{"gigabit", "kibibyte", "122070.3125"},
	{"gigabit", "pebibyte", "0.0000001110223"},
	{"gigabit", "yobibyte", "0.0000000000000001033976"},
	{"gigabit", "exbibyte", "0.0000000001084202"},
	{"gigabit", "bit", "1000000000"},
	{"gigabit", "nibble", "250000000"},
	{"gigabit", "megabit", "1000"},
	{"gigabit", "zebibyte", "0.0000000000001058791"},
	{"gigabit", "byte", "125000000"},
	{"gigabit", "petabit", "0.000001"},
	{"gigabit", "mebibyte", "119.20929"},
	{"gigabit", "exabit", "0.000000001"},
	{"gigabit", "terabit", "0.001"},
	{"gigabit", "Tebibyte", "0.0001136868"},
	{"gigabit", "gibibyte", "0.1164153"},
	{"gigabit", "kilobit", "1000000"},
	{"bit", "kibibyte", "0.0001220703"},
	{"bit", "pebibyte", "0.0000000000000001110223"},
	{"bit", "yobibyte", "0.0000000000000000000000001033976"},
	{"bit", "exbibyte", "0.0000000000000000001084202"},
	{"bit", "gigabit", "0.000000001"},
	{"bit", "nibble", "0.25"},
	{"bit", "megabit", "0.000001"},
	{"bit", "zebibyte", "0.0000000000000000000001058791"},
	{"bit", "byte", "0.125"},
	{"bit", "petabit", "0.000000000000001"},
	{"bit", "mebibyte", "0.0000001192093"},
	{"bit", "exabit", "0.000000000000000001"},
	{"bit", "terabit", "0.000000000001"},
	{"bit", "Tebibyte", "0.0000000000001136868"},
	{"bit", "gibibyte", "0.0000000001164153"},
	{"bit", "kilobit", "0.001"},
	{"nibble", "kibibyte", "0.000488281"},
	{"nibble", "pebibyte", "0.000000000000000444089"},
	{"nibble", "yobibyte", "0.00000000000000000000000041359"},
	{"nibble", "exbibyte", "0.000000000000000000433681"},
	{"nibble", "gigabit", "0.000000004"},
	{"nibble", "bit", "4"},
	{"nibble", "megabit", "0.000004"},
	{"nibble", "zebibyte", "0.000000000000000000000423516"},
	{"nibble", "byte", "0.5"},
	{"nibble", "petabit", "0.000000000000004"},
	{"nibble", "mebibyte", "0.000000476837"},
	{"nibble", "exabit", "0.000000000000000004"},
	{"nibble", "terabit", "0.000000000004"},
	{"nibble", "Tebibyte", "0.000000000000454747"},
	{"nibble", "gibibyte", "0.000000000465661"},
	{"nibble", "kilobit", "0.004"},
	{"megabit", "kibibyte", "122.070312"},
	{"megabit", "pebibyte", "0.0000000001110223"},
	{"megabit", "yobibyte", "0.0000000000000000001033976"},
	{"megabit", "exbibyte", "0.0000000000001084202"},
	{"megabit", "gigabit", "0.001"},
	{"megabit", "bit", "1000000"},
	{"megabit", "nibble", "250000"},
	{"megabit", "zebibyte", "0.0000000000000001058791"},
	{"megabit", "byte", "125000"},
	{"megabit", "petabit", "0.000000001"},
	{"megabit", "mebibyte", "0.1192093"},
	{"megabit", "exabit", "0.000000000001"},
	{"megabit", "terabit", "0.000001"},
	{"megabit", "Tebibyte", "0.0000001136868"},
	{"megabit", "gibibyte", "0.0001164153"},
	{"megabit", "kilobit", "1000"},
	{"zebibyte", "kibibyte", "1152921504606846976"},
	{"zebibyte", "pebibyte", "1048576"},
	{"zebibyte", "yobibyte", "0.000976562"},
	{"zebibyte", "exbibyte", "1024"},
	{"zebibyte", "gigabit", "9444732965739.291016"},
	{"zebibyte", "bit", "9444732965739290427392"},
	{"zebibyte", "nibble", "2361183241434822606848"},
	{"zebibyte", "megabit", "9444732965739290"},
	{"zebibyte", "byte", "1180591620717411303424"},
	{"zebibyte", "petabit", "9444732.965739"},
	{"zebibyte", "mebibyte", "1125899906842624"},
	{"zebibyte", "exabit", "9444.732966"},
	{"zebibyte", "terabit", "9444732965.73929"},
	{"zebibyte", "Tebibyte", "1073741824"},
	{"zebibyte", "gibibyte", "1099511627776"},
	{"zebibyte", "kilobit", "9444732965739290624"},
	{"byte", "kibibyte", "0.000976562"},
	{"byte", "pebibyte", "0.000000000000000888178"},
	{"byte", "yobibyte", "0.000000000000000000000000827181"},
	{"byte", "exbibyte", "0.000000000000000000867362"},
	{"byte", "gigabit", "0.000000008"},
	{"byte", "bit", "8"},
	{"byte", "nibble", "2"},
	{"byte", "megabit", "0.000008"},
	{"byte", "zebibyte", "0.000000000000000000000847033"},
	{"byte", "petabit", "0.000000000000008"},
	{"byte", "mebibyte", "0.000000953674"},
	{"byte", "exabit", "0.000000000000000008"},
	{"byte", "terabit", "0.000000000008"},
	{"byte", "Tebibyte", "0.000000000000909495"},
	{"byte", "gibibyte", "0.000000000931323"},
	{"byte", "kilobit", "0.008"},
	{"petabit", "kibibyte", "122070312500"},
	{"petabit", "pebibyte", "0.1110223"},
	{"petabit", "yobibyte", "0.0000000001033976"},
	{"petabit", "exbibyte", "0.0001084202"},
	{"petabit", "gigabit", "1000000"},
	{"petabit", "bit", "1000000000000000"},
	{"petabit", "nibble", "250000000000000"},
	{"petabit", "megabit", "1000000000"},
	{"petabit", "zebibyte", "0.0000001058791"},
	{"petabit", "byte", "125000000000000"},
	{"petabit", "mebibyte", "119209289.550781"},
	{"petabit", "exabit", "0.001"},
	{"petabit", "terabit", "1000"},
	{"petabit", "Tebibyte", "113.686838"},
	{"petabit", "gibibyte", "116415.321827"},
	{"petabit", "kilobit", "1000000000000"},
	{"mebibyte", "kibibyte", "1024"},
	{"mebibyte", "pebibyte", "0.000000000931323"},
	{"mebibyte", "yobibyte", "0.000000000000000000867362"},
	{"mebibyte", "exbibyte", "0.000000000000909495"},
	{"mebibyte", "gigabit", "0.00838861"},
	{"mebibyte", "bit", "8388608"},
	{"mebibyte", "nibble", "2097152"},
	{"mebibyte", "megabit", "8.388608"},
	{"mebibyte", "zebibyte", "0.000000000000000888178"},
	{"mebibyte", "byte", "1048576"},
	{"mebibyte", "petabit", "0.00000000838861"},
	{"mebibyte", "exabit", "0.00000000000838861"},
	{"mebibyte", "terabit", "0.00000838861"},
	{"mebibyte", "Tebibyte", "0.000000953674"},
	{"mebibyte", "gibibyte", "0.000976562"},
	{"mebibyte", "kilobit", "8388.608"},
	{"exabit", "kibibyte", "122070312500000"},
	{"exabit", "pebibyte", "111.022302"},
	{"exabit", "yobibyte", "0.0000001033976"},
	{"exabit", "exbibyte", "0.1084202"},
	{"exabit", "gigabit", "1000000000"},
	{"exabit", "bit", "1000000000000000000"},
	{"exabit", "nibble", "250000000000000000"},
	{"exabit", "megabit", "1000000000000"},
	{"exabit", "zebibyte", "0.0001058791"},
	{"exabit", "byte", "125000000000000000"},
	{"exabit", "petabit", "1000"},
	{"exabit", "mebibyte", "119209289550.78125"},
	{"exabit", "terabit", "1000000"},
	{"exabit", "Tebibyte", "113686.837722"},
	{"exabit", "gibibyte", "116415321.826935"},
	{"exabit", "kilobit", "1000000000000000"},
	{"terabit", "kibibyte", "122070312.5"},
	{"terabit", "pebibyte", "0.0001110223"},
	{"terabit", "yobibyte", "0.0000000000001033976"},
	{"terabit", "exbibyte", "0.0000001084202"},
	{"terabit", "gigabit", "1000"},
	{"terabit", "bit", "1000000000000"},
	{"terabit", "nibble", "250000000000"},
	{"terabit", "megabit", "1000000"},
	{"terabit", "zebibyte", "0.0000000001058791"},
	{"terabit", "byte", "125000000000"},
	{"terabit", "petabit", "0.001"},
	{"terabit", "mebibyte", "119209.289551"},
	{"terabit", "exabit", "0.000001"},
	{"terabit", "Tebibyte", "0.1136868"},
	{"terabit", "gibibyte", "116.415322"},
	{"terabit", "kilobit", "1000000000"},
	{"Tebibyte", "kibibyte", "1073741824"},
	{"Tebibyte", "pebibyte", "0.000976562"},
	{"Tebibyte", "yobibyte", "0.000000000000909495"},
	{"Tebibyte", "exbibyte", "0.000000953674"},
	{"Tebibyte", "gigabit", "8796.093022"},
	{"Tebibyte", "bit", "8796093022208"},
	{"Tebibyte", "nibble", "2199023255552"},
	{"Tebibyte", "megabit", "8796093.022208"},
	{"Tebibyte", "zebibyte", "0.000000000931323"},
	{"Tebibyte", "byte", "1099511627776"},
	{"Tebibyte", "petabit", "0.00879609"},
	{"Tebibyte", "mebibyte", "1048576"},
	{"Tebibyte", "exabit", "0.00000879609"},
	{"Tebibyte", "terabit", "8.796093"},
	{"Tebibyte", "gibibyte", "1024"},
	{"Tebibyte", "kilobit", "8796093022.208"},
	{"gibibyte", "kibibyte", "1048576"},
	{"gibibyte", "pebibyte", "0.000000953674"},
	{"gibibyte", "yobibyte", "0.000000000000000888178"},
	{"gibibyte", "exbibyte", "0.000000000931323"},
	{"gibibyte", "gigabit", "8.589935"},
	{"gibibyte", "bit", "8589934592"},
	{"gibibyte", "nibble", "2147483648"},
	{"gibibyte", "megabit", "8589.934592"},
	{"gibibyte", "zebibyte", "0.000000000000909495"},
	{"gibibyte", "byte", "1073741824"},
	{"gibibyte", "petabit", "0.00000858993"},
	{"gibibyte", "mebibyte", "1024"},
	{"gibibyte", "exabit", "0.00000000858993"},
	{"gibibyte", "terabit", "0.00858993"},
	{"gibibyte", "Tebibyte", "0.000976562"},
	{"gibibyte", "kilobit", "8589934.592"},
	{"kilobit", "kibibyte", "0.1220703"},
	{"kilobit", "pebibyte", "0.0000000000001110223"},
	{"kilobit", "yobibyte", "0.0000000000000000000001033976"},
	{"kilobit", "exbibyte", "0.0000000000000001084202"},
	{"kilobit", "gigabit", "0.000001"},
	{"kilobit", "bit", "1000"},
	{"kilobit", "nibble", "250"},
	{"kilobit", "megabit", "0.001"},
	{"kilobit", "zebibyte", "0.0000000000000000001058791"},
	{"kilobit", "byte", "125"},
	{"kilobit", "petabit", "0.000000000001"},
	{"kilobit", "mebibyte", "0.0001192093"},
	{"kilobit", "exabit", "0.000000000000001"},
	{"kilobit", "terabit", "0.000000001"},
	{"kilobit", "Tebibyte", "0.0000000001136868"},
	{"kilobit", "gibibyte", "0.0000001164153"},
	{"fahrenheit", "kelvin", "255.927778"},
	{"fahrenheit", "celsius", "-17.222222"},
	{"kelvin", "fahrenheit", "-457.87"},
	{"kelvin", "celsius", "-272.15"},
	{"celsius", "fahrenheit", "33.8"},
	{"celsius", "kelvin", "274.15"},
}

func testConversions(t *testing.T, convTests []conversionTest) {
	fmtOpts := FmtOptions{false, false, 6}
	for _, cTest := range convTests {
		label := fmt.Sprintf("%s <-> %s conversion", cTest.from, cTest.to)
		t.Run(label, func(t *testing.T) {
			u1, err := Find(cTest.from)
			if err != nil {
				t.Fatal(err.Error())
			}
			u2, err := Find(cTest.to)
			if err != nil {
				t.Fatal(err.Error())
			}
			res := MustConvertFloat(1.0, u1, u2)
			assert.Equal(t, cTest.val, res.Fmt(fmtOpts),
				"%s -> %s conversion test failed", cTest.from, cTest.to)

			// test inverse conversion
			ires := MustConvertFloat(res.Float(), u2, u1)
			assert.Equal(t, 1.0, roundFloat(ires.Float(), 12),
				"%s <- %s conversion test failed", cTest.from, cTest.to)
		})
	}
}

func roundFloat(f float64, precision uint) float64 {
	r := math.Pow(10, float64(precision))
	return math.Round(f*r) / r
}

func TestConversionValues(t *testing.T) {
	testConversions(t, convTests)
}
